;*************T************************************************
; File Name		:"CFdriver.asm"
; Title			: Compact-Flash driver for AVR (Atmega)
; Date			: 27-01-2001
; Version		: 1.3
; Author		: Loek Gijben
; Contact               : L.H.Gijben@Flatnet.TUDelft.nl
;
;  	Compact-Flash IO module for Atmel AVR microcontrollers.
;	This AVR-assembler code contains:
;	-Some macro's for reducing sourcecode.
;	-WriteATAreg	:Write 1 byte to an ATA register.
;	-ReadATAreg	:Read a single byte from an ATA register.
;	-ReadCFSectors	:Read (x-times) 512 bytes from CF to (X)SRAM.	
;	-WriteCFSectors	:Write (x-times) 512 bytes from (X)SRAM to CF.
;	-Identify_Drive	:Identify drive.
;	-Format		;Format sector(s).
;	-Sleep		;Save energy by sending CF to bed.
; 	-RESET		;Soft reset (but I always just pull the plug).
;
;	Microcontroller general purpose registers used in all routines:
;	- tempCF1-3		; (r18-20)
;	- ATAdata		; (r21)
;	- ATAaddress		; (r22)
;	- CF_Wr_LowR/CF_Rd_LowR	; (r0)
;	- CF_Wr_HighR/CF_Rd_LowR; (r1)
;	- X
;***************************************************************

;***************************************************************
;	Ten macros to make source smaller and readable.
;  (To make final binary code smaller you may convert them to
;  subroutines, at cost of just a little speed)
; -Two flexible but slower macros to read/write data/command to/from
;  any ATA register.
; -Two macros to set the datadirection of the databus (input/output).
; -A macro (+alternative) for waiting for the CF ready signal.
; -Four speed optimized macros for sector transfers.
;***************************************************************

; Macro to write data in ATAdata
; to flash on ATA register low nibble(ATAaddress) in 8 clockcycles. 
.macro WByte
	out ATAdataPort, ATAdata	; Put data on bus
	out ATAaddressPort, ATAaddress	; Address setup + CE 
	cbi ATAaddressPort, CF_notWr    ; then WE
	sbi ATAaddressPort, CF_notWr	; -WE clocks in the byte
	sbi ATAaddressPort, CF_notCE	; and finally chip disable CF
.endmacro

; macro to read the byte in ATA register low(ATAaddress)
; into ATAdata in 9 clockcycles.
.macro RByte
	out ATAaddressPort, ATAaddress	; Setup address + CE.
	cbi ATAaddressPort, CF_notRd	; .. Rd/OE
	nop				; The old 40MB PCMCIA flashdisk needs a
					; waitstate, remove this for real CF.
	in ATAdata, ATAdataINPort	; (read data on PINA).
	sbi ATAaddressPort, CF_notRd	; -Rd
	sbi ATAaddressPort, CF_notCE	; -CE
.endmacro

; Alternative macro that makes use of the fact that data will still be available
; on the pins of ATAdataPort *after* the Rd strobe.
;.macro RByte	
;	out ATAaddressPort, ATAaddress	; Setup address + CE.
;	cbi ATAaddressPort, CF_notRd		; .. Rd/OE
;	sbi ATAaddressPort, CF_notRd		; -Rd
;	in ATAdata, ATAdataINPort	; (read still available into r19)
;	sbi ATAaddressPort, CF_notCE		; -CE
;.endmacro

; Two macros to set the datadirection on ATAdataPort
.macro	putRd
	push tempCF3
	ldi tempCF3, $00
	out ATAdataPort, tempCF3	; Prepare PortA for tri-state
	out ATAdatadir, tempCF3		; Make sure that PortA is input 
	pop tempCF3
.endmacro

.macro	putWr
	push tempCF3
	ldi tempCF3, $FF
	out ATAdatadir, tempCF3		; Make sure that PortA becomes output 
	pop tempCF3
.endmacro

;Macro to wait for RDY after a command has been issued.
.macro BSYLoop	
BSYLoop:
	sbis CF_RDYBSYPort, CF_RDYBSY	; Check RDY/BSY 
	jmp BSYLoop			; until pin is driven high again by CF
.endmacro

; Alternative macro to get rid of BSY/RDY signal line (But you must realize
; that you must use the ATA A3 address-line, which you also may drop otherwise)
; It makes use of BUSY bit in Alternate status ATA register. According to the
; specs reading the status register will clear the signal, while reading  the 
; alternate status register won't.
;.macro BSYloop
;	push ATAaddress
;	push ATAdata
;	ldi ATAaddress, $6E		; Setup Alternate Status register address + CE
;BSYLoopPoll:
;	wdr
;	out ATAaddressPort, ATAaddress	; Setup address + CE.
;	cbi ATAaddressPort, CF_notRd	; .. Rd/OE
;	sbi ATAaddressPort, CF_notRd	; -Rd
;	in ATAdata, ATAdataINPort	; (read still available into r19)
;	sbi ATAaddressPort, CF_notCE	; -CE
;	sbrc ATAdata, 7			; See if Busy bit is set
;	rjmp BSYLoopPoll		; else continue looping
;	ldi ATAaddress, $60		; Back to ATA data register.
;	pop ATAdata
;	pop ATAaddress
;.endmacro 

; Optimized macros for sector transfers. When external SRAM is not used, or the
; ports for XSRAM access don't coincide with those for CF access, then the
; special code where the external RAM interface is switched on/off may be 
; omitted. 
;.macro Inbyte	
;		out ATAaddressPort, CF_Rd_LowR	; Set pin -Rd high in one tick
;		nop				; Waitstate for old model 40 MB PC-card
;		in ATAdata, ATAdataINPort	; 
;		out ATAaddressPort, CF_Rd_HighR	; 
;		st  X+, ATAdata			; Store data (8 clockcycles)
;.endmacro

.macro Inbyte	
		out ATAaddressPort, CF_Rd_LowR	; Set pin -Rd high in one tick
		out ATAaddressPort, CF_Rd_HighR	; 
		in ATAdata, ATAdataINPort	; Data will still be on the pins..
		st  X+, ATAdata			; Set data in SRAM 
.endmacro					; (5 clockcycles)

.macro XInbyte					; Read byte to external SRAM, 
		out ATAaddressPort, CF_Rd_LowR	; Set pin -Rd high in one tick
		out ATAaddressPort, CF_Rd_HighR	; 
		in ATAdata, ATAdataINPort	; 
		out MCUCR, tempCF1		; Expects tempCF1 to hold value $80
						; enabling the XSRAM
		st  X+, ATAdata			; Store data (8 clockcycles)
		out MCUCR, tempCF2		; Expected to be $00: disable XSRAM	 
.endmacro					; (8 clockcycles)

.macro Outbyte
		ld ATAdata, X+			;
		out ATAdataPort, ATAdata	; Put data from X= on databus.
		out ATAaddressPort, CF_Wr_LowR  ; Strobe the -Wr signal
		;nop				; Uncomment for waitstate (AVR > 8 MHz)
		out ATAaddressPort, CF_Wr_HighR	; Byte clocked in in 5 cycles
.endmacro
.macro XOutbyte					; Write byte from external SRAM, 
		out MCUCR, tempCF1		; Enable XSRAM
		ld ATAdata, X+			;
		out MCUCR, tempCF2		; Disable XSRAM again.	 
		out ATAdataPort, ATAdata	; Put data from X= on databus.
		out ATAaddressPort, CF_Wr_LowR  ; Strobe the -Wr signal
		;nop				; Uncomment for Waitstate (AVR > 6.7 MHz)
		out ATAaddressPort, CF_Wr_HighR	; Byte clocked in in 5 cycles
.endmacro



;***************************************************************
; 	-WriteATAreg
; 	Write one byte from ATAdata to ATA register
;
; 	Parameters: 
;	-ATA register number in low nibble ATAaddress
;	-Data/command to write in ATAdata		  
;***************************************************************

WriteATAreg:	
	putWr
	WByte			; A little bit trivial?
	ret			; 

;***************************************************************
; 	-ReadATAreg
; 	Read one byte from ATA register to ATAdata
;
; 	Parameters: 
;	-ATA register number in low nibble ATAaddress
;	-Data will end up in ATAdata		  
;***************************************************************

ReadATAreg:	
	putRd
	RByte
	ret 

;***************************************************************
; 	-ReadCFSectors
; 	Read one or more sectors from CF register to (X)SRAM
;
; 	Parameters: 
;	-CF Sector Count register must be set, and kept in 
;	 tempCF3.
;	-CF LBA registers must be set
;	-Data will end up in (X)SRAM in a contiguous block
;	 starting from the value in AVR X-register 		  
;
;	When external SRAM is not used, or the CF is not on any
;	port that is also in use by XSRAM all code that tests for
;	XSRAM addresses or switches on/off SRE in MCUCR may be 
;	left out.
;
;	Routine has three parts:1) Issuing the actual command and
;				   waiting for RDY
;				2) Checking for SRAM or XSRAM use
;				3) Clocking in x-times 512 bytes
;
;***************************************************************

ReadCFSectors:				; Part one.
	ldi ATAdata, CF_RD_Sectors	;
	ldi ATAaddress, CF_Cmd_High	; Prepare to write a command
	rcall WriteATAreg		; and execute it.
					; While we wait for RDY do
InSector:				; some housekeeping. 
	PutRd				; Switch databus to input.
	ldi tempCF1, CF_Rd_Low		; Load CF_Rd_Low/High registers.
	mov CF_Rd_LowR, tempCF1
	ldi tempCF1, CF_Rd_High
	mov CF_Rd_HighR, tempCF1
	ldi tempCF1, XSRAMon		; Fill tempCF1 and tempCF2
	ldi tempCF2, XSRAMoff		; with XSRAMon/off.
	BSYloop				; Wait for Compact-Flash RDY.

Rd_SCloop:		; Part two 		
			; With each sector transfer we have to check 
			; whether we will end up in XSRAM regions.
	cpi XH, $0E	; So check X against (RAMEND-512 = $0DFF)
	brlo Rd_SRAM	; $0DFF or lower will certainly stay in SRAM 
	brne X_Rdjmp	; $0E00 or higher must be in XSRAM
	tst XL		; Test for start address $0E00 (XL=$00)
	breq Rd_SRAM	; 'cause that's still in SRAM
X_Rdjmp:
	rjmp Rd_XSRAM 	; and above $0E00 ends in XSRAM

Rd_SRAM:		; Part three(SRAM)
	push tempCF3
	ldi tempCF3, 33	
Rd_Transferloop:	
	wdr
	Inbyte		; About the same loop as for XSRAM 
	Inbyte
	Inbyte
	Inbyte
	Inbyte
	Inbyte
	Inbyte
	Inbyte
	Inbyte
	Inbyte
	Inbyte
	Inbyte
	Inbyte
	Inbyte
	Inbyte
	Inbyte
	dec tempCF3		; Decrement the loopcounter.	
	breq DoneInbyte		; 512 bytes read?
	rjmp Rd_Transferloop	; If not do another 16
DoneInbyte: 
	rjmp Rd_SCtest	; Go see if there are more sectors to transfer.

Rd_XSRAM:		; Part three(XSRAM)
	push tempCF3	; Save the SectorCount, and make it
	ldi tempCF3, 33 ; Loopcounter (for 32 times)
Rd_XTransferloop:	
	wdr
	XInbyte		; Each test-rjmp routine takes 4 cycles overhead
	XInbyte		; per byte transfer = 40-80%. A thing to do is to
	XInbyte		; roll out the loop: An ATmega has plenty of
	XInbyte		; codespace...
	XInbyte		; The first steps of rolling out count the most:
	XInbyte		; 4 bytes per loop decrease the overhead to 
	XInbyte		; about 5-10%
	XInbyte		; 16 bytes per loop reduce it to 1-2%
	XInbyte		; rolling out all 512 bytes takes up 5-30kB and makes
	XInbyte		; the code very long (not to mention dull)
	XInbyte		;
	XInbyte		; This driver compromises in 16 bytes per loop.
	XInbyte
	XInbyte
	XInbyte
	XInbyte
	dec tempCF3	; 32 x 16 bytes makes one sector read
	breq DoneXInbyte; If tempCF3 = zero then we're done 
	rjmp Rd_XTransferloop	; else do another 16 bytes. 
DoneXInbyte: 
	;rjmp Rd_SCtest	; Go see if there are more sectors to transfer.

Rd_SCtest:
	pop tempCF3	; Get back the Sectorcounter.
	dec tempCF3
	breq End_Rdloop	; Do another sector transfer if not zero
	rjmp Rd_SCloop
End_Rdloop:		; else end the read sectors loop.
	ldi tempCF3, CF_Off	; Unselect CF (-CE high).
	out ATAaddressPort, tempCF3	 		
	ret		; End routine.

;***************************************************************
; 	-WriteCFSectors
; 	Write one or more sectors from (X)SRAM register to CF
;
; 	Parameters: 
;	-CF Sector Count register must be set, and kept in 
;	 tempCF3.
;	-CF LBA registers must be set
;	-Data will be read from a contiguous block (X)SRAM  
;	 starting from the value in AVR X-register 		  
;
;	When external SRAM is not used, or the CF is not on any
;	port that is also in use by XSRAM all code that tests for
;	XSRAM addresses or switches on/off SRE in MCUCR may be 
;	left out.
;
;	Routine has three parts:1) Issuing the actual command and
;				   waiting for RDY
;				2) Checking for SRAM or XSRAM use
;				3) Clocking out x-times 512 bytes
;
;***************************************************************
WriteCFSectors:				; Part one
	ldi ATAdata, CF_Wr_Sectors	;
	ldi ATAaddress, CF_Wr_High	; Prepare a write command
	rcall WriteATAreg		; and execute it
					; While we wait for RDY do some housekeeping  
	
	ldi tempCF1, CF_Wr_Low		; Load CF_Wr_Low/High registers.
	mov CF_Wr_LowR, tempCF1
	ldi tempCF1, CF_Wr_High
	mov CF_Wr_HighR, tempCF1
	ldi tempCF1, XSRAMon		; Fill tempCF1/2 with enable/disable XSRAM.
	ldi tempCF2, XSRAMoff		; 
	;PutRd				; Uncomment when using alternative BSYloop
	BSYloop				; Wait for RDY.
	;PutWr				; Uncomment when using the alternative BSYloop.

Wr_SCloop:		; Part two 		
			; With each sector transfer we have to check 
			; whether we will end up in XSRAM regions
	cpi XH, $0E	; So check X against (RAMEND-512 = $0E00)
	brlo Wr_SRAM	; $0DFF or lower will certainly stay in SRAM 
	brne X_Wrjmp	; $0E00 or higher must be in XSRAM
	tst XL		; Test for start address $0E00 (XL=$00)
	breq Wr_SRAM	; 'cause that's still in SRAM
X_Wrjmp:
	rjmp Wr_XSRAM 	; and above $0D00 ends in XSRAM

Wr_SRAM:		; Part three(SRAM)
	push tempCF3
	ldi tempCF3, 33

Wr_Transferloop:	
	Outbyte		; About the same loop as for XSRAM 
	Outbyte
	Outbyte
	Outbyte
	Outbyte
	Outbyte
	Outbyte
	Outbyte
	Outbyte
	Outbyte
	Outbyte
	Outbyte
	Outbyte
	Outbyte
	Outbyte
	Outbyte
	dec tempCF3	
	breq DoneOutbyte	; 512 bytes read?
	rjmp Wr_Transferloop
DoneOutbyte:
	rjmp Wr_SCtest	; Go see if there are more sectors to transfer

Wr_XSRAM:		; Part three(XSRAM).
	push tempCF3	; Save the SectorCount
	ldi tempCF3, 33	; Loopcounter (tempCF1 +tempCF2 already in use)
Wr_XTransferloop:	
	XOutbyte	; Please see Rd_XSRAM for comments on this loop
	XOutbyte
	XOutbyte
	XOutbyte
	XOutbyte
	XOutbyte
	XOutbyte
	XOutbyte
	XOutbyte
	XOutbyte
	XOutbyte
	XOutbyte
	XOutbyte
	XOutbyte
	XOutbyte
	XOutbyte
	dec tempCF3	;32 x 16 bytes is one sector read
	breq DoneXOutbyte
	rjmp Wr_XTransferloop
DoneXOutbyte:	
	;rjmp Wr_SCtest	; Go see if there are more sectors to transfer

Wr_SCtest: 		; Find out if there are more sectors to transfer
	pop tempCF3
	dec tempCF3	; Remember this one still contained <Sector Count>?
	breq End_Wrloop	; Do another sector transfer if not zero
	rjmp Wr_SCloop
End_Wrloop:
	ldi tempCF3, CF_Off	; Unselect CF (-CE high).
	out ATAaddressPort, tempCF3	 		
	ret		; End routine.



;***************************************************************
; 	-Identify Drive
; 	Read a special "sector" with drive related information
;
; 	Parameters: 
;	-512 bytes data will end up in (X)SRAM 
; 	according starting point set by X-register
;		  
;***************************************************************

Identify_Drive:
	ldi ATAdata, CF_ID_DRV		; Set ID_DRV command ready
	ldi ATAaddress, CF_CMD_High	; 
	rcall WriteATAreg		; and write it to command register
	ldi tempCF3, $01		; Needed for the readloop (1 sector read)
	rcall InSector			; and read in the data 
	ret

;***************************************************************
; 	-Format
; 	Format sector(s) (Fill sectors with $FF pattern)
;
; 	Parameters: 
;	-Sector Count must be set, and kept in tempCF3
;	-LBA must be set in ATAregisters 2-6 
; 	-This command needs (x-times) 512 bytes written to the 
;	 dataregister after issuing the command (like with a
;	 Write Sector(s) command.
;		  
;***************************************************************

Format:
	ldi ATAdata, CF_Format		; Set ID_DRV command ready
	ldi ATAaddress, CF_CMD_High	; 
	rcall WriteATAreg		; and write it to command register
	BSYLoop
	call WR_SRAM			; A format needs to have sectors written to it
	ret

;***************************************************************
; 	-Sleep
; 	Put the CF in Standby immediate
;	This command is to save power. Now the CF can be send to
; 	bed within 20 microseconds, if omitted the CF will enter
;	sleep mode automatically after 5ms.
;
; 	Parameters: none 
;***************************************************************

Sleep:
	ldi ATAdata, CF_Sleep		; Set Sleep command ready
	ldi ATAaddress, CF_CMD_High	; 
	rcall WriteATAreg		; and write it to command register
	ret

;***************************************************************
; 	-Reset
; 	Does a soft reset, but only if it accepts commands.
;	This command is merely for compatibility reasons with
; 	other types of ATA devices.
;
; 	Parameters: none 
; (this command does not work well with my old PC-card, it stays
;  in reset condition :^( )
;***************************************************************

RESET:
	ldi ATAdata, CF_Reset		; Set SW_Rst bit
	ldi ATAaddress, CF_DevCont	; .. in Device Control register
	rcall WriteATAreg		; and write it to this register,
	ldi ATAdata, CF_Reset		; release the reset button...
	rcall WriteATAreg		; and write it to this register.
	ret
	
;***************************************************************

;************* End of CF_driver.asm ****************************
