; 
; File Name		:"VFD.asm"
; Title			:VFD routines
; Date			:
; Version		:
; Support telephone	:765 287 1987  David B. VanHorn
; Support fax		:765 287 1989
; Support Email		:dvanhorn@cedar.net
; Target MCU		:AT90S8515
;
; DESCRIPTION
; Written for a Futaba US162SD03CB display
; Two lines of 16 chars each, Sync serial interface. Only 3 pins, and FAST!
;
;***************************************************************************;
;	M O D I F I C A T I O N   H I S T O R Y 
;
;
;       rev.      date    who   why
;	----	--------  ---	------------------------------------------
;	0.01	98.11.14  dvh	Creation
;	0.02	99.09.10  dvh	Cleanup, made a distinct buffer separate
;				from the LCD. Also cleanups to make it compatible
;				with the new buffer managers. All working
;	0.03	99.09.15  dvh	Added Put_VFD to allow char-by char entry into
;				the VFD buffer with auto-scrolling
;	0.04	99.09.17  dvh	Investigating corruption at high update rates
;
;***************************************************************************
;Pin
;number	Symbol	Function
;    1	VCC
;    2	Clock	
;    3	Ground
;    4	Data
;    5	Reset
;
;************************************************************
;
;Hardware pins assigned here to logical names.
;
.equ	VFD_DATA=PORTA	;VFD data lines interface
.equ	VFD_DDR=DDRA	;
.equ	VFD_R=5		;VFD Reset
.equ	VFD_D=6		;VFD Data
.equ	VFD_C=7		;VFD Clock
;
;*************************************************************
;You will notice a certain similarity between this and the LCD 
;code. This is a port of the LCD code, since it contained all 
;the basic functions. Only the lowest level driver and the names
;of the buffers have been changed to protect the innocent.
;*************************************************************
;Things you need to set up:
;
;You have to define things, before you can use them.
;Below, in the CSEG (Code Segment) I define some constants.
;Then, in the DSEG (Data segment) I use them to allocate some memory.
;If you reverse this order, the assembler may not pick up an error,
;But you will experience "non-deterministic execution".
;
;
;VFD variables

.CSEG
.equ	VFD_Lines=2
.equ	VFD_LEN=16
.equ	VFD_OUT_Size=VFD_Lines * VFD_Len
.equ	VFD_Scroll_Speed = 100	;ms per action, used with VFD_Timer
;
.DSEG
VFD_OUT_BUF:	.byte	VFD_OUT_Size+2	;Size defined above
VFD_Timer:	.byte	1	;A timer in mS between actions like scrolling
VFD_BITFLAGS:	.byte 1		;7=
				;6= 1-Display Dirty, needs updating from buffer
				;5=
				;4=
				;3=
				;2=
				;1=
				;0=

.CSEG

;*************************************************************
;
;A byte called "VFD_Bitflags". I use bit 6 (x1xxxxxx) to flag
;that the VFD_BUF has changed, and the display needs updating.
;An equate for the display size
;Set a buffer somewhere in RAM, called VFD_BUF, whose size
;is VFD_Size
;See EQUATES.INC for samples, but you can redefine if needed.
;
;In the idle routine, test VFD_Bitflags. If bit 6 is set, then call
;VFD_Spew to send the buffer to the display. It will clear bit
;6 so the test fails next time. This cuts down on EMI from
;excessive display updates. It also frees up a lot of CPU time.
;
;You can also call the following scroll routines:
;Scroll_All_VFD_Left	;Entire display left
;Scroll_All_VFD_Right	;Entire display right
;Scroll_Top_VFD_Left	;Top half left
;Scroll_Top_VFD_Right	;Top half right
;Scroll_Bot_VFD_Left	;Bottom half left
;Scroll_Bot_VFD_Right	;Bottom half right
;
;The scroll routines only affect the buffer, but they also 
;set the flag in Bitflags so that the display will be updated.
;
;          00000000011111111112222222222333
;          12345678901234567890123456789012
VFD_TEST:
.db       "Getting Started  Version FOUR.0 ",0
;
;************************************************************
;Prompt routines. To start with, just call this one
;************************************************************
;
Say_VFD_Test:
	ldi	ZL,low(VFD_TEST*2)	;Make the Z reg point at the table
	ldi	ZH,high(VFD_TEST*2)	;preparing for the LPM instruction
	rcall	VFD_STR_OUT		;String to buf, buf to VFD
	ret	
;
;************************************************************
;Places a char on the VFD display, at the last open position.
;If the display is full, then it scrolls all left, to make room
;************************************************************
;
;
Put_VFD:
	;Check how much room there is.
	ldi	YH,high(VFD_OUT_BUF)	;Point at the beginning
	ldi	YL,low(VFD_OUT_BUF)	;
	ld	TEMP3,Y			;This gets the current length
	ldd	TEMP2,Y+1		;This gets the total size
	cp	TEMP2,TEMP3		;
	brne	Put_VFD_End_Char	;If not full, then just add to
	push	TEMP			;
	rcall	Kill_Head		;
	pop	TEMP			;	

Put_VFD_End_Char:
	;Put the char in as the last char
	ldi	YH,high(VFD_OUT_BUF)	;Point at the beginning
	ldi	YL,low(VFD_OUT_BUF)	;
	rcall	Store_Buffer		;

	lds	TEMP_BITFLAGS,VFD_Bitflags;
	ori	TEMP_Bitflags,$40	;Flag the display dirty
	sts	VFD_Bitflags,TEMP_Bitflags
	ret

;
;************************************************************
;VFD Scrolling based on time.
;************************************************************
;
;You might want to split this up into something like:
; TIMED_VFD_SCROLL_TOP_LEFT etc. so that you can call them individually.
;
;
VFD_Scroll:
	lds	Temp,VFD_Timer		;Get the system timer (dec'd by T0)
	and	TEMP,TEMP		;Check it
	brne	VFD_Scroll_Done		;If non-zero, don't bother
	ldi	TEMP,VFD_Scroll_Speed	;Otherwise reload the timer first
	sts	VFD_Timer,TEMP		;so the task time dosen't bother
					;the planned repition rate

	;Now perform scrolling as desired

	;rcall	Scroll_All_VFD_Left	;Don't turn on conflicting pairs
	rcall Scroll_Top_VFD_Left	;like top right and top left, or
	;rcall Scroll_Bot_VFD_Left	;the result won't be interesting.
	;rcall	Scroll_All_VFD_Right	;It won't hurt anything though
	;rcall Scroll_Top_VFD_Right	;
	rcall Scroll_Bot_VFD_Right	;

VFD_Scroll_Done:
	ret

;***************************************************************************
;Update the display, only if needed
;***************************************************************************
;
Check_VFD_Display:
	lds	TEMP,VFD_Bitflags
	andi	TEMP,$40	;Remove all but the "Dirty Bit"
	breq	CK_VFD_Disp_Done;If zero, then just exit
	rcall	VFD_Spew	;Otherwise, update the display

	lds	TEMP,VFD_Bitflags
	andi	TEMP,$BF	;Reset "dirty bit"
	sts	VFD_BitFlags,TEMP

CK_VFD_Disp_Done:
	ret			;
;****************************************************************
;
;These routines roll the entire display memory from left to right, or right to left.
;
Scroll_All_VFD_Left:
	push	TEMP			;
	push	TEMP2			;
	push	LOOP			;

	ldi	ZH,high(VFD_OUT_BUF)	;Point at the beginning
	ldi	ZL,low(VFD_OUT_BUF)	;
	ld	TEMP,Z+			;This gets the current length
	ld	TEMP,Z+			;This gets the total size
	mov	LOOP,TEMP		;
	mov	YL,ZL			;
	mov	YH,ZH			;
	ld	TEMP,Z+			;Now pointing at the second byte
	ld	TEMP2,Y			;Now holding the first byte
	rjmp	Scroll_VFD_Left_Loop	;

Scroll_Top_VFD_Left:
	push	TEMP			;
	push	TEMP2			;
	push	LOOP			;
	
	ldi	ZH,high(VFD_OUT_BUF)	;Point at the beginning
	ldi	ZL,low(VFD_OUT_BUF)	;
	ld	TEMP,Z+			;Get current length
	ld	TEMP,Z+			;Get max size
	clc				;
	ror	TEMP			;Half that
	mov	LOOP,TEMP		;How many to shift.

	mov	YL,ZL			;
	mov	YH,ZH			;
	ld	TEMP2,Z+		;Pick up the first one
	;At this point, Z points to buffer+1, and Y points to buffer.
	;Temp2 contains the first char, which will be over-written by
	;the first shift.
	rjmp	Scroll_VFD_Left_Loop	;

Scroll_Bot_VFD_Left:
	push	TEMP			;
	push	TEMP2			;
	push	LOOP			;

	ldi	ZH,high(VFD_OUT_BUF)	;Point at the beginning
	ldi	ZL,low(VFD_OUT_BUF)	;
	ld	TEMP,Z+			;
	ld	TEMP,Z+			;now have size
	clc				;
	ror	TEMP			;
	mov	LOOP,TEMP		;Save half of that

	clc				;
	add	ZL,TEMP			;16 bit pointer add
	adc	ZH,ZERO			;

	mov	YL,ZL			;
	mov	YH,ZH			;
	ld	TEMP2,Z+		;Pick up the first one
	rjmp	Scroll_VFD_Left_Loop	;
;
;The Y and Z pointers are set, and TEMP2 contains the end char, TEMP has the 
;number of chars to scroll. All we do here is pick them up, and set them down.
;
Scroll_VFD_Left_Loop:

	ld	TEMP,Z+			;Pick up the second char and then point at 
					;the third
	st	Y+,TEMP			;Store second at first, and then point at 
					;the second.	
	dec	Loop			;One less to shift.
	brne	Scroll_VFD_Left_Loop	;
	
	ld	TEMP,-Y			;Adjust, temp is junk
	st	Y,TEMP2			;Put the first char in the last cell
	lds	TEMP_BITFLAGS,VFD_Bitflags;
	ori	TEMP_Bitflags,$40	;Flag the display dirty
	sts	VFD_Bitflags,TEMP_Bitflags
	pop	LOOP			;
	pop	TEMP2			;
	pop	TEMP			;
	ret



Scroll_All_VFD_Right:
	push	TEMP			;
	push	TEMP2			;
	push	LOOP			;


	ldi	ZH,high(VFD_OUT_BUF)	;
	ldi	ZL,low(VFD_OUT_BUF)	;
	ld	TEMP,Z+			;Get size
	ld	TEMP,Z+			;Get length
	mov	LOOP,TEMP		;
	dec	TEMP			;
	clc				;
	add	ZL,TEMP			;16 bit add
	adc	ZH,Zero			;
	mov	YH,ZH			;Copy pointers
	mov	YL,ZL			;

	ld	TEMP2,Z			;Pick up the first one
	ld	R0,-Z			;Post dec
	rjmp	Scroll_VFD_Right_Loop	;

Scroll_Top_VFD_Right:
	push	TEMP			;
	push	TEMP2			;
	push	LOOP			;

	ldi	ZH,high(VFD_OUT_BUF)	;Point at the beginning
	ldi	ZL,low(VFD_OUT_BUF)	;
	ld	TEMP,Z+			;Skip the size
	ld	TEMP,Z+			;Get the length
	clc				;

	ror	TEMP			;
	mov	LOOP,TEMP		;
	dec	TEMP			;Tweak
	clc				;
	add	ZL,TEMP			;
	adc	ZH,ZERO			;
	mov	YL,ZL			;
	mov	YH,ZH			;

	ld	TEMP2,Z			;Pick up the first one
	ld	R0,-Z			;And back one
	rjmp	Scroll_VFD_Right_Loop	;

Scroll_Bot_VFD_Right:
	push	TEMP			;
	push	TEMP2			;
	push	LOOP			;

	ldi	ZH,high(VFD_OUT_BUF)	;Point at the beginning
	ldi	ZL,low(VFD_OUT_BUF)	;
	ld	TEMP,Z+			;Skip the size
	ld	TEMP,Z+			;Get the length
	dec	TEMP			;
	clc				;
	add	ZL,TEMP			;
	adc	ZH,ZERO			;
	mov	YL,ZL			;
	mov	YH,ZH			;

	inc	TEMP			;
	clc				;
	ror	TEMP			;Half that
	mov	LOOP,TEMP		;

	ld	TEMP2,Z				;Pick up the first one
	ld	R0,-Z				;Toss
	rjmp	Scroll_VFD_Right_Loop		;
;
;Same as Scroll_Left_Loop, but note the pre-dec rather than post inc.
;
Scroll_VFD_Right_Loop:

	ld	TEMP,Z			;Pick it up here
	ld	R0,-Z			;This just does a post dec on Z
	st	Y,TEMP			;Put it down there
	ld	R0,-Y			;And a post dec on Y
	dec	Loop			;One less to mess with
	brne	Scroll_VFD_Right_Loop	;If not done, do more
	
	ld	TEMP,Y+			;Tweak
	st	Y,TEMP2			;Put the first char in the last cell
	lds	TEMP_Bitflags,VFD_Bitflags;
	ori	TEMP_Bitflags,$40	;Flag the display dirty
	sts	VFD_Bitflags,TEMP_Bitflags

	pop	LOOP			;
	pop	TEMP2			;
	pop	TEMP			;

	ret
;****************************************************************
;Called with Z pointing to a string in rom.
;Seeks and displays forward, until we find a null.
;Shoves that text into the VFD_OUT_BUF
;****************************************************************	
;
VFD_STR_NOPAD:

	push	TEMP		;
	push	TEMP2		;
	push	LOOP		;
	rcall	VFD_STR2BUF	;Move a string in rom to VFD_OUT_BUF
	lds	TEMP_BITFLAGS,VFD_Bitflags	;
	ori	TEMP_Bitflags,$40	;Flag the display dirty
	sts	VFD_Bitflags,TEMP_Bitflags
	pop	LOOP		;
	pop	TEMP2		;
	pop	TEMP		;
	ret

VFD_STR_OUT:

	push	TEMP		;
	push	TEMP2		;
	push	LOOP		;
	rcall	VFD_STR2BUF	;Move a string in rom to VFD_OUT_BUF
	rcall	VFD_Fill	;
	lds	TEMP_Bitflags,VFD_Bitflags;
	ori	TEMP_Bitflags,$40;Flag the display dirty
	sts	VFD_Bitflags,TEMP_Bitflags
	pop	LOOP		;
	pop	TEMP2		;
	pop	TEMP		;
	ret
;
;****************************************************************
;This routine moves a string of arbitrary size into VFD_OUT_BUF,
;and padds it with spaces at the end as needed.
;****************************************************************
;
VFD_STR2BUF:
	;Point at beginning of VFD_OUT_BUF
	ldi	YH,high(VFD_OUT_BUF)	;
	ldi	YL,low(VFD_OUT_BUF)	;
	rcall	String_Out		;
	ret

	ld	TEMP2,Y+		;Get the length
	ld	TEMP3,Y+		;Get the size

	;No need to check for full, this is a replace

VFD_STR_Loop:
	lpm			;look up character
	ld	TEMP,Z+		;Move pointer to in string
	mov	TEMP,R0		;move to temp
	and	TEMP,TEMP	;Is it null?
	breq	VFD_STR_Done	;Yes, we're done, see if we need to fill

	dec	LOOP		;Keep track of how many we've stored
	breq	VFD_STR_DONE	;No need to fill, can't do anymore anyway

	st	Y+,TEMP		;Output to ram buffer, inc after
	rjmp	VFD_STR_Loop	;

VFD_STR_Done:
	ret
;
;************************************************************
;
VFD_Fill:	;Now the VFD_OUT_BUF has the output data, but may contain junk

	ldi	YH,high(VFD_OUT_BUF)	;
	ldi	YL,low(VFD_OUT_BUF)	;
	ld	TEMP2,Y+		;Get the length
	ld	TEMP3,Y+		;Get the size
	sub	TEMP3,TEMP2		;
	breq	VFD_Fill_Done	;If full, then we're ready to spew

VFD_Fill_Loop:
	ldi	TEMP,$20	;(Space)
	st	Y+,TEMP		;
	dec	TEMP3		;
	breq	VFD_Fill_Loop	;

VFD_Fill_Done:
	ret
;
;**************************************************************
;This routine moves the data in VFD_OUT_BUF to the display.
;Called only from IDLE, this updates the display to be current 
;with the contents of the VFD buffer. 
;**************************************************************
;
VFD_SPEW:
	rcall	VFD_Clear		;Clear and home.
	;
	;Point at the beginning of the VFD buffer
	ldi	YH,high(VFD_OUT_BUF)	;
	ldi	YL,low(VFD_OUT_BUF)	;
	ld	TEMP2,Y+		;Get length
	cpi	TEMP2,0			;Nothing left?
	breq	VFD_Spew_Done		;Then exit, all done
	mov	LOOP,TEMP2		;
	ld	TEMP3,Y+		;Get size (not used)

VFD_Spew_Loop:
	ld	TEMP,Y+			;Get a char, and inc Y
	rcall	VFD_Talk		;Send it to the display
	dec	LOOP
	brne	VFD_Spew_Loop		;If not done, do it again


VFD_Spew_Done:
	lds	TEMP_Bitflags,VFD_Bitflags;
	andi	TEMP_Bitflags,$BF	;Flag the display clean
	sts	VFD_Bitflags,TEMP_Bitflags
	ret				;
;
;
;*************************************************************
;The main powerup init routine. Only called after powerup
;*************************************************************
;
VFD_INIT:
	sbi	VFD_DDR,VFD_C	;Make them outputs
	sbi	VFD_DDR,VFD_R	;
	sbi	VFD_DDR,VFD_D	;

	sbi	VFD_Data,VFD_C	;
	cbi	VFD_Data,VFD_R	;
	sbi	VFD_Data,VFD_D	;
	rcall	VFD_Reset	;(Takes a while!)
	rcall	VFD_Clear	;

	rcall	VFD_Flash_Off	;

	ldi	TEMP,$30	;Set flash rate
	rcall	VFD_Flash_Rate	;

	ldi	TEMP,$FF	;Max bright
	rcall	VFD_Bright	;

	ldi	TEMP,$01	;Select Western font
	rcall	VFD_Font	;
	ret
;
;*************************************************************
;Unlike most of these routines, this one takes two parms.
;Start of range in TEMP (1-32) end of range in TEMP2 (1-32)
;*************************************************************
;
VFD_Flash_Range:
	sbi	VFD_Data,VFD_C	;
	cbi	VFD_Data,VFD_R	;
	sbi	VFD_Data,VFD_D	;
	push	TEMP		;Save the start byte
	rcall	VFD_Talk	;Send the command
	pop	TEMP		;Get the start byte back
	rcall	VFD_Talk	;Send it
	mov	TEMP,TEMP2	;Get the end byte
	rcall	VFD_Talk	;Send it
	ret
;
;*************************************************************
;Turn on the flashing
;*************************************************************
;
VFD_Flash_On:
	sbi	VFD_Data,VFD_C	;
	cbi	VFD_Data,VFD_R	;
	sbi	VFD_Data,VFD_D	;
	ldi	TEMP,$07	;
	rcall	VFD_Talk	;
	ldi	TEMP,$02	;
	rcall	VFD_Talk	;
	ret
;
;*************************************************************
;Reset the display
;*************************************************************
;
VFD_Reset:
	sbi	VFD_Data,VFD_C	;Assure the initial conditions
	cbi	VFD_Data,VFD_R	;
	sbi	VFD_Data,VFD_D	;
	ldi	TEMP,255	;Wait a bit
	rcall	MS_Delay	;
	sbi	VFD_Data,VFD_R	;Assert reset
	ldi	TEMP,1		;Wait a bit
	rcall	MS_Delay	;
	cbi	VFD_Data,VFD_R	;De-Assert reset
	ldi	TEMP,3		;Wait a bit
	rcall	MS_Delay	;
	sbi	VFD_Data,VFD_C	;Assure exit conditions
	cbi	VFD_Data,VFD_R	;
	sbi	VFD_Data,VFD_D	;
	ret			;
;
;*************************************************************
;Turn off the flashing
;*************************************************************
;
VFD_Flash_Off:
	sbi	VFD_Data,VFD_C	;
	cbi	VFD_Data,VFD_R	;
	sbi	VFD_Data,VFD_D	;
	ldi	TEMP,$07	;
	rcall	VFD_Talk	;
	ldi	TEMP,$01	;
	rcall	VFD_Talk	;
	ret
;
;*************************************************************
;Set the flash rate, remember to set a zone for flashing
;*************************************************************
;
VFD_Flash_Rate:
	sbi	VFD_Data,VFD_C	;
	cbi	VFD_Data,VFD_R	;
	sbi	VFD_Data,VFD_D	;
	Push	TEMP		;
	ldi	TEMP,$08	;
	rcall	VFD_Talk	;
	pop	TEMP		;
	rcall	VFD_Talk	;
	ret
;
;*************************************************************
;Select english or Japaneese font.
;*************************************************************
;
VFD_Font:
	sbi	VFD_Data,VFD_C	;
	cbi	VFD_Data,VFD_R	;
	sbi	VFD_Data,VFD_D	;
	push	TEMP		;Save the brightness
	ldi	TEMP,$09	;The command
	rcall	VFD_Talk	;Send it
	pop	TEMP		;
	rcall	VFD_Talk	;
	ret			;
;
;*************************************************************
;Sets the display brightness
;*************************************************************
;
VFD_Bright:
	sbi	VFD_Data,VFD_C	;Assure the initial conditions
	cbi	VFD_Data,VFD_R	;
	sbi	VFD_Data,VFD_D	;
	push	TEMP		;Save the brightness
	ldi	TEMP,$04	;The command
	rcall	VFD_Talk	;Send it
	pop	TEMP		;Get the brightness back
	rcall	VFD_Talk	;Send it
	ret			;
;
;*************************************************************
;Clear and home the display
;*************************************************************
;
VFD_CLEAR:

	sbi	VFD_Data,VFD_C	;Assure the initial conditions
	cbi	VFD_Data,VFD_R	;
	sbi	VFD_Data,VFD_D	;
        ldi	TEMP,$01	;The command itself
        rcall	VFD_Talk	;
	ret
;
;**************************************************************
;Sends character to VFD
;Required character must be in TEMP
;**************************************************************
;
VFD_Talk:
	sbi	VFD_Data,VFD_C	;Assure the startup conditions
	cbi	VFD_Data,VFD_R	;
	sbi	VFD_Data,VFD_D	;

	push	LOOP		;I need to borrow these
	push	TEMP		;

	ldi	TEMP,8		;How many bits to send
	mov	LOOP,TEMP	;Put it in LOOP
	pop	TEMP		;Put temp back

VFDT_Loop:
	rol	TEMP		;Don't know which direction yet.
	brcc	VFDT_Zero	;If it's a zero, then send zero
	sbi	VFD_Data,VFD_D	;Otherwise it's a one
	rjmp	VFDT_Clk	;Clock it out

VFDT_Zero:
	cbi	VFD_Data,VFD_D	;Simple, isn't it!

VFDT_Clk:
	rcall	VFD_Clock	;Whang the clock line
	dec	LOOP		;One less to send
	brne	VFDT_Loop	;If not done, then 

	pop	LOOP	
	sbi	VFD_Data,VFD_C	;
	cbi	VFD_Data,VFD_R	;
	sbi	VFD_Data,VFD_D	;
	ret			
;
;*************************************************************
;This measures out at slightly longer than the spec timing,
;but when I tried it at 2uS over the spec, it had intermittent
;problems with spurious chars on the display.
;*************************************************************
;
VFD_Clock:
	rcall	TwouS		;Required setup time
	rcall	TwouS		;
	cbi	VFD_Data,VFD_C	;
	rcall	TwouS		;Required pulse width
	rcall	TwouS		;
	sbi	VFD_Data,VFD_C	;				
	rcall	TwouS		;Required hold time and
	rcall	TwouS		;cycle time
	rcall	TwouS		;
	rcall	TwouS		;
	rcall	TwouS		;
	rcall	TwouS		;
	rcall	TwouS		;
	rcall	TwouS		;
	rcall	TwouS		;
	rcall	TwouS		;
	rcall	TwouS		;
	ret			;
;
;*************************************************************
;Brute force delay, 2uS at 8 MHz
;*************************************************************
;
TwouS:
	nop	;Two uS delay, plus a bit
	nop	;
	nop	;
	nop	;I really despise this sort of thing,
	nop	;but sometimes, you just can't avoid it.
	nop	;
	nop	;
	nop	;
	
	nop	;
	nop	;
	nop	;
	nop	;
	nop	;
	nop	;
	nop	;
	nop	;
	ret	;
;
;************************************************************
