;***************************************************************************
; 
; File Name		:'STEPPER.asm"
; Title			:
; Date			:
; Version		:
; Support telephone	:765 287 1987  David B. VanHorn
; Support fax		:765 287 1989
; Support Email		:dvanhorn@cedar.net
; Target MCU		:AT90S8515
; 
;***************************************************************************;
;	M O D I F I C A T I O N   H I S T O R Y 
;
;
;       rev.      date    who   why
;	----	--------  ---	------------------------------------------
;	0.01	98.08.30  dvh	Creation
;	0.02	98.09.03  dvh	More info on how to electrically drive steppers
;	0.03	99.09.11  dvh	Made the output configurable to any port,
;				and the bits are not not required to be 
;				sequential on that port.
;********************************************************************
;
;
;A simple routine to control a stepper motor. 
;Full or half-step, outputs designed for unipolar motors.
;Outputs on any port, on any four bits
;
;Full stepping
;
;Phase	Current
;
;0000	1	(off)
;1000	1	1st state
;0100	1
;0010	1
;0001	1	4th state
;
;Half-Stepping
;
;Phase  Current
;
;0000	0	(off)
;1000	1	1st state
;1100	0
;0100	1
;0110	0
;0010	1
;0011	0
;0001	1
;1001	0	8th state
;
;
;	ULN2003 driver IC  (2803 is similar, but one more driver, and 2 more pins)
;
;     Inputs   Outputs
;
;	1 ----> 16  High input = low output (coil active)
;	2 ----> 15
;	3 ----> 14	
;	4 ----> 13	
;	5 ----> 12	
;	6 ----> 11	
;	7 ----> 10	
;   GND	8	9 Diode 
;
;
;
;Each driver sinks current when it's input is at a logic high. Each driver has a 
;protection diode, all diodes come to pin 9, which should be connected to the 
;stepper's +V supply.
;For this application, connect the AVR stepper output pins to pins 1,2,3,4 of 
;the 2003, pin 8 of the IC to the development board's ground, and 16,15,14,13 
;of the 2003 to the stepper's coil leads. Connect the two common leads of the
;stepper to pin 9, and apply a modest voltage (3-6V) from an external power supply. 
;Be sure to tie the external supply ground to the development board ground.
;
;This IC uses the stepper's coil resistance to limit the applied current, so be sure
;that your stepper supply voltage does not exceed the current rating of the chip, or 
;the thermal limits of the motor. Rare earth magnet steppers can be permanently "cooked"
;by applying too much power for too long.  The chip's own voltage limit is 50V, but the
;curent limit and thermal limits are much more important.  The IC is rated for 500mA max.
;
;A good treatment on stepper motors is available at:
;http://www.cs.uiowa.edu/~jones/step/circuits.html
;
;Going forward, step up the table, going backward, step back :)
;Up to 1mS per step (or half-step) with a simple control routine based on the 1mS ISR
;
;Ram variables
;
;Step_Dir	Forward or reverse (0=Reverse, 1=forward, anything else= no step)?
;Step_Time	mS per phase (0-255)
;Step_State	Where are we now? A pointer into the stepper output table
;Step_Mode	Full or half step (0=full 1=half)
;Step_Speed	Reload value for Step_Time. Your application can change this and 
;		alter the stepper speed. Beware, steppers don't like large changes
;		in velocity.
;
;*********************************************************************
;
.DSEG
Step_Dir:	.byte	1		;Flag, Forward or reverse direction?
Step_Time:	.byte	1		;Is it time to step?
Step_State:	.byte	1		;Where are we now?
Step_Mode:	.byte	1		;Flag, Full or half step?
Step_Rate:	.byte	1		;mS per step
.CSEG
;
;*********************************************************************
;
;Currently the code demands all the bits on the same port.
;With a little more work, you could concievable make it take
;any bit on many ports, but I'm not sure that's useful.
;
.equ	Step_Out=PORTD	;Which port you want your bits on
.equ	Step_DDR=DDRD	;This must be the same as the port above
.equ	Step_PH1=3	;I didn't have to use sequential bits.
.equ	Step_PH2=4	;but it was convenient.
.equ	Step_PH3=5	;You could also use bits 0,3,5,7 if it 
.equ	Step_PH4=6	;amused you to do so.
.equ	Step_Default = 50		;mS per step (used only in INIT.ASM)
;
;
;I assume external hardware will take care of current control in half step mode
;
; BIG HAIRY NOTE!  Because of the way that the assembler works, and
;the fact that EPROM is 16 bit words, tables built as follows are 
;actually allocating one word per byte, Have a look at the read routine
;in STEPPER.ASM for these tables as opposed to the one above. Both 
;work, but putting more than one byte per .db statement results in
;more efficient use of the romspace.
;
Full_Step_Table:
.db	$08	;1000 Full current
.db	$04	;0100 Full current
.db	$02	;0010 Full current
.db	$01	;0001 Full current

Half_Step_Table:
.db	$08	;1000 Full current
.db	$0C	;1100 Half current
.db	$04	;0100 Full current
.db	$06	;0110 Half current
.db	$02	;0010 Full current
.db	$03	;0011 Half current
.db	$01	;0001 Full current
.db	$09	;1001 Half current
;
;
;*******************************************************************
;
;Initial states for stepper motor control
;
Init_Stepper:
	ldi	TEMP,$00		;
	out	Step_Out,TEMP		;Turn off the stepper outputs.
					;and VFD signals

	ldi	TEMP,$00		;
	sts	Step_State,TEMP		;Initial state for the stepper motor

;
;Use only one direction at a time. NOTE: I am initializing to stopped, so you'll need to 
;set a direction as shown below before anything is going to happen
;
	ldi	TEMP,$01		;
	sts	Step_Dir,TEMP		;Set forward as default direction.

	;ldi	TEMP,$00		;
	;sts	Step_Dir,TEMP		;Set reverse as default direction.

	;ldi	TEMP,$02		;
	;sts	Step_Dir,TEMP		;Set stopped as default direction.

	ldi	TEMP,$01
	sts	Step_Mode,TEMP		;Set half step as default mode.

	;ldi	TEMP,$00
	;sts	Step_Mode,TEMP		;Set full step as default mode.

	ldi	TEMP,Step_Default	;Default step rate			;
	sts	Step_Rate,TEMP		;Establish a step rate
	sts	Step_Time,TEMP		;and set it running

	ret

;*********************************************************************
;
Step_Motor: ;First, see if we're stepping this time
	lds	TEMP,Step_Time	;Is it time to step?
	cpi	TEMP,$00	;
	breq	Step_Direction	;If so, then check direction, else
	ret			;just bail


Step_Direction: ;Then determine the direction
	lds	TEMP,Step_Dir	;Get the current direction
	and	TEMP,TEMP	;
	breq	Step_Reverse	;
	cpi	TEMP,$01	;Is it forward?
	breq	Step_Forward	;Go if so.
	rjmp	Step_Done	;else it's no step


Step_Forward: ;Then wether full or half (forward)
	lds	TEMP,Step_Mode	;Full or half
	and	TEMP,TEMP	;
	breq	Step_FF		;
	rjmp	Step_FH		;

Step_Reverse: ;Or wether full or half (reverse)
	lds	TEMP,Step_Mode	;Full or half?
	and	TEMP,TEMP	;
	breq	Step_RF		;
	rjmp	Step_RH		;
;
;***************************************************************************************
;
;Step forard, a full step
;
Step_FF:
	lds	TEMP,Step_State			;Get the current state
	inc	TEMP				;advance one step on the full step table
	cpi	TEMP,$04			;
	brne	Step_FFA			;
	ldi	TEMP,$00			;
Step_FFA:
	ldi	ZL,low(Full_Step_Table*2)	;Make the Z reg point at the table
	ldi	ZH,high(Full_Step_Table*2)	;preparing for the LPM instruction
	rjmp	Step_Output			;
;
;Step forward, a half step
;
Step_FH:
	lds	TEMP,Step_State			;Get the current state
	inc	TEMP				;advance one step on the full step table
	cpi	TEMP,$08			;
	brne	Step_FHA			;
	ldi	TEMP,$00			;
Step_FHA:
	ldi	ZL,low(Half_Step_Table*2)	;Make the Z reg point at the table
	ldi	ZH,high(Half_Step_Table*2)	;preparing for the LPM instruction
	rjmp	Step_Output			;
;
;Step reverse a full step
;
Step_RF:
	lds	TEMP,Step_State			;Get the current state
	dec	TEMP				;advance one step on the full step table
	cpi	TEMP,$FF			;
	brne	Step_RFA			;
	ldi	TEMP,$03			;

Step_RFA:
	ldi	ZL,low(Full_Step_Table*2)	;Make the Z reg point at the table
	ldi	ZH,high(Full_Step_Table*2)	;preparing for the LPM instruction
	rjmp	Step_Output			;
;
;Step reverse a half step
;
Step_RH:
	lds	TEMP,Step_State			;Get the current state
	dec	TEMP				;advance one step on the full step table
	cpi	TEMP,$FF			;
	brne	Step_RHA			;
	ldi	TEMP,$07			;
Step_RHA:
	ldi	ZL,low(Half_Step_Table*2)	;Make the Z reg point at the table
	ldi	ZH,high(Half_Step_Table*2)	;preparing for the LPM instruction
	rjmp	Step_Output			;
;
;Whichever and how far, output the motor states
;
Step_Output:
	sts	Step_State,TEMP			;
	lsl	TEMP				;Mult x 2, because we store words in rom
						;as separate .DB statements rather than 
						;as a long string.
	clc					;
	add	ZL,TEMP				;Add the calculated offset
	adc	ZH,ZERO				;

STO_A:	lpm					;look up character
	mov	TEMP,R0				;
	andi	TEMP,$0F			;Mask off non-stepper bits
	sbi	Step_DDR,Step_PH1		;Make the bits outputs
	sbi	Step_DDR,Step_PH2		;
	sbi	Step_DDR,Step_PH3		;
	sbi	Step_DDR,Step_PH4		;

	;Take the low nybble and use it to turn the step
	;bits on and off as appropriate

STO_D1:
	ror	TEMP		;
	brcs	STO_D1H		;	
	cbi	Step_Out,Step_Ph1;
	rjmp	STO_D2		;	
STO_D1H:
	sbi	Step_Out,Step_PH1;

STO_D2:
	ror	TEMP		;
	brcs	STO_D2H		;	
	cbi	Step_Out,Step_Ph2;
	rjmp	STO_D3		;	
STO_D2H:
	sbi	Step_Out,Step_PH2;

STO_D3:
	ror	TEMP		;
	brcs	STO_D3H		;	
	cbi	Step_Out,Step_Ph3;
	rjmp	STO_D4		;	
STO_D3H:
	sbi	Step_Out,Step_PH3;

STO_D4:
	ror	TEMP		;
	brcs	STO_D4H		;	
	cbi	Step_Out,Step_Ph4;
	rjmp	Step_Done	;	
STO_D4H:
	sbi	Step_Out,Step_PH4;


Step_Done:

	lds	TEMP,Step_Rate			;Restart the timer for the next step
	sts	Step_Time,TEMP			;
	ret					;


