;***************************************************************************
; 
; File Name		:'MEMORY.asm"
; Title			:
; Date			:
; Version		:
; Support telephone	:765 287 1987  David B. VanHorn
; Support fax		:765 287 1989
; Support Email		:dvanhorn@cedar.net
; Target MCU		:AT90S8515
;
; DESCRIPTION
;
; All routines that put data in or out of memory buffers
; 
;***************************************************************************;
;	M O D I F I C A T I O N   H I S T O R Y 
;
;       rev.      date    who   why
;	----	--------  ---	------------------------------------------
;	0.01	98.07.29  dvh	Creation
;	0.02	98.08.28  dvh	Regularized to XL,XH notation instead of R26.R27 etc
;	0.03	98.09.01  dvh	STRING_SEROUT now enables UDRIE, so we start transmitting.
;				The UDRIE ISR will turn it off when we're done talking.
;	0.04	98.10.19  dvh	Fixed a stack bug that would crash you if the serout buffer became
;				full.
;	0.05	99.05.27  dvh   Constructing generalized buffer manager routines,
;				called by pointing to the desired buffer in Y, and
;				they all work the same. Note: LPM uses Z, so Z dosen't
;				work as a pointer to the destination.
;
;				New linear buffer structure:
;				LSBBBBBBBBBBBBBBBBBBBBBB
;				L is the length byte, and is located AT the 
;				buffer address. The buffer starts at address
;				+2, and runs to address + len + 2
;				If L = 0, then the buffer is empty.
;				S is the buffer size, if L=S then the buffer is 
;				full.
;
;				All FIFO buffers use a routine for "put", and another 
;				for "take". 
;
;	0.06	99.09.09  dvh	Added TEST_SPACE which is used when stuffing a string
;				into a buffer. It returns a zero if the destination
;				buffer is too full to accept the string.
;				The calling routine can then abort or whatever
;				based on that.
;
;	0.07	99.09.13  dvh	Removed serial buffer, superceded by circular
;				buffer manager in SERIAL.ASM
;
;	0.08	99.09.15  dvh	Added low level circular buffer managers
;
;	0.09	99.09.23  dvh	Circ_Read and Circ_Scan now working.
;				Read allows you to copy data without eating it.
;				Scan returns the presence and position of a byte
;				within the circ buffer. Useful for "wait till you
;				get an X" type protocols. 
;				Until you get the "X", you may want to eat the head
;				of the buffer to make room, or not, that's up to you.
;
;***************************************************************************
;
;Each linear buffer looks like this:
;<S><L>DATA
;S is the current size of the buffer (in this case 4)
;L is the length of the buffer, which could be anything.
;If L-S=0 then the buffer is full.
;The length does not change during run-time, size is dependent on the 
;contents, can be anywhere from zero to equal to size.
;Initially, they are set to zero size, and whatever length is set up
;in EQUATES.INC. 
;
INIT_Buffers:
	
	cli				;Turn off all ints for a moment
	rcall	Clean_Ram		;Clean up the ram, and flag buffers
	rcall	Flag_Ram		;Note that this does NOT result in
					;buffers set to arbitrary size,
					;because of the following routines
					;that explicitly set the buffer
					;lengths and sizes.

	ldi	YH,high(Morse_Out_Buf)	;Get the buffer head
	ldi	YL,low(Morse_Out_Buf)	;
	ldi	TEMP,0			;I'm empty
	st	Y+,TEMP			;
	ldi	TEMP,(Morse_Size)	;and I can be this big
	st	Y,TEMP			;

	ldi	YH,high(LCD_Out_Buf)	;Get the buffer head
	ldi	YL,low(LCD_Out_Buf)	;
	ldi	TEMP,0			;
	st	Y+,TEMP			;
	ldi	TEMP,(LCD_out_Size)	;
	st	Y,TEMP			;

	ldi	YH,high(VFD_Out_Buf)	;Get the buffer head
	ldi	YL,low(VFD_Out_Buf)	;
	ldi	TEMP,0			;
	st	Y+,TEMP			;
	ldi	TEMP,(VFD_out_Size)	;
	st	Y,TEMP			;

	sei				;Let the interrupts fly!
	ret				;
;
;************************************************************
;
;Used for strings that must fit into the dest buffer.
;Z points to the string, and Y to the buffer.
;
Test_Space:
	push	ZH			;Save the original pointer
	push	ZL			;
	push	YH			;
	push	YL			;
	ldi	TEMP,0			;
Test_Space_Loop:
	lpm				;Get the byte into R0 and incr Z
	and	R0,R0			;Is it null
	breq	Test_Space_Check	;Yes, all done
	ld	R0,Z+			;inc pointer
	inc	TEMP			;inc byte counter
	rjmp	Test_Space_Loop		;Again.

Test_Space_Check:
	ld	TEMP2,Y+		;Get current length
	ld	TEMP3,Y			;and size
	sub	TEMP3,TEMP2		;Size - contents = freespace 
	cp	TEMP3,TEMP		;CP Freespace to string len
	brlo	Test_Space_Full		;

Test_Space_NotFull:
	ldi	TEMP,1			;
	and	TEMP,TEMP		;
	rjmp	Test_Space_Exit		;

Test_Space_Full:			;
	ldi	TEMP,0			;
	and	TEMP,TEMP		;
	;rjmp	Test_Space_Exit		;

Test_Space_Exit:
	pop	YL			;
	pop	YH			;
	pop	ZL			;
	pop	ZH			;
	ret				;
;
;************************************************************
;
;This routine takes a null terminated string, pointed to by Z, 
;and places it in the buffer pointed to by Y
;Will place as much of the string as will fit into the dest, and no more.
;The calling routine should have used Test_Space if it matters.
;
String_out:
	lpm			;Get the byte into R0 and incr Z
	and	R0,R0		;Is it null
	breq	STR_out_Exit	;Yes, all done
	mov	TEMP,R0		;
	rcall	Store_Buffer	;Take this byte and shove it
	ld	TEMP,Z+		;Inc Z, toss the data.
	rjmp	String_Out	;

Str_Out_Exit:
	ret
;
;
;************************************************************
;
;Stores a byte in a buffer pointed to by YH,YL
;Data is in TEMP
;Trashes TEMP2,TEMP3
;
Store_Buffer:

	;First check that the buffer can take more.
	push	YH		;First, save that pointer to the buffer
	push	YL		;We'll need it later, that's where the length is.

	;Note, these two leave y pointing past the length and size, and at the 
	;location where the data would go, if the buffer was empty.

	ld	TEMP2,Y+	;Get the current length
	ld	TEMP3,Y+	;Get the current size

	cp	TEMP2,TEMP3	;Are they equal?
	breq	Store_Buffer_Done;Yes, then just exit, can't take more

Store_Buf_Data:
	add	YL,TEMP2		;Add the length
	brcc	Store_buf_Data_B	;This is a 16 bit add, so the buffer can
	inc	YH			;live anywhere in ram

Store_buf_Data_B:
	st	Y,TEMP			;Store the data
	inc	TEMP2			;Inc the lenght

Store_Buffer_Done:
	pop	YL			;Get that pointer back
	pop	YH			;
	st	Y,TEMP2			;store the length

	ret				;pau!
;
;*************************************************************
;
;Used to delete a single char off the head of a linear buffer.
;Point YH,YL to beginning of buffer Head
;Returns junk in TEMP and TEMP2
;

Kill_Head:

	ld	TEMP2,Y		;Get current length
	cpi	TEMP2,0		;If no string, then
	breq	Kill_Head_Done	;there's nothing to do

	;YH,YL working pair pointing at head of the data
	push	TEMP2		;
	push	YH		;
	push	YL		;
	adiw	YL,2		;Point at the data
	mov	XH,YH		;
	mov	XL,YL		;
	adiw	XL,1		;Point X at next cell

Kill_Head_Loop:
	ld	TEMP,X+		;Shuffle down
	st	Y+,TEMP		;
	dec	TEMP2		;
	brne	Kill_Head_Loop	;

	pop	YL		;Get the original address
	pop	YH		;back into Y
	pop	TEMP2		;Get the original length
	dec	TEMP2		;-1
	st	Y,TEMP2		;store it

Kill_Head_Done:
	ret			;

;
;*************************************************************
;
;Used to delete a single char off the tail of a linear buffer
;Point YH,YL to beginning of buffer
;Returns junk in temp
;
Kill_Tail:
	ld	TEMP,Y		;Get the length
	cpi	TEMP,0		;Is it zero?
	breq	Kill_Tail_Done	;If so, we're done.
	dec	TEMP		;else make it shorter
	st	Y,TEMP		;and store it back.
Kill_Tail_Done:
	ret			;
;
;**********************************************************************************
;Debug only, stores distinctive data into each linear buffer, making it easy to see them
;in the simulator.
;
Flag_Ram:
;	


F_3:	
	ldi	ZH,high(Morse_OUT_BUF)	;
	ldi	ZL,low(Morse_OUT_BUF)	;
	ldi	TEMP,$44		;
	ldi	TEMP2,Morse_SIZE	;
	rcall	Fill_Buf		;

F_4:	
	ldi	ZH,high(LCD_OUT_BUF)	;
	ldi	ZL,low(LCD_OUT_BUF)	;
	ldi	TEMP,$45		;
	ldi	TEMP2,LCD_OUT_SIZE	;
	rcall	Fill_Buf		;

F_5:	
	ldi	ZH,high(VFD_OUT_BUF)	;
	ldi	ZL,low(VFD_OUT_BUF)	;
	ldi	TEMP,$46		;
	ldi	TEMP2,VFD_OUT_SIZE	;
	rcall	Fill_Buf		;

F_Fin:	
	ret				;Done.

;
;Fill a given buffer to a given size with 
;given data.
;
Fill_Buf:
	st	Z+,TEMP			;Fill a byte
	dec	TEMP2			;One less to fill
	brne	Fill_Buf		;Are we done?
	ret				;yes, exit
;
;***************************************************************************
;
;This routine flushes any previous junk out of our workspace.
;Avoids any pattern-sensitivity or programmer headspace errors with
;uninitialized variables.
;
Clean_Ram:
	ldi	ZH,$00			;Point @ $0060
	ldi	ZL,$60			;
	clr	TEMP			;

CleanLoop:
	st	Z+,TEMP			;Store $00 at each location
	cpi	ZH,$02			;If not at least here, then go do it again
	brne	CleanLoop		;
	cpi	ZL,$5C			;DONT EAT THE STACK!
	brne	CleanLoop		;Getting close!
	ret				;Done.

;
;*************************************************************
;
;R28,29 are buffer tail pointer R30,31 are buffer head
;Return number of chars stored between R28,30
;Zero flag on exit says no more 
;
Check_It:
	ld	TEMP2,Y+	;Get the size
	ld	TEMP,Y		;Get the length
	sbc	TEMP,TEMP2	;If the result is zero,
	ret			;then no more will go.
;
;***************************************************************************************
;
;Generic circular buffer store and recall routines.
;NOTE that the ISRs use their own routines, to prevent ints corrupting
;the data while both background and foreground are manipulating the buffers
;These routines unconditionally disable ints, and re-enable at the end
;
;Using Y as the buffer pointer, pointing to Tail in Put
;and head in Get.
;
;Returns a byte in TEMP, if carry is set, then the byte is valid in the 
;get case, and in the put case, the byte was stored 
;
;In order to make these routines easily callable, we have to
;have some ground rules.
;
;Circular Buffers are always less than 256 bytes.
;The buffer address is NOT the beginning of the active buffer area, 
;instead, it points to the size byte. 
;SHT(Data)
;Size,Head,Tail,Data(of a constant length)
;On calling these routines, TEMP2 must hold the buffer's length (constant)
;and temp has data to be stored, or will contain data read back.
;On return, carry indicates sucess, and TEMP2 has the current size of the 
;buffer. TEMP has returned data, if any


Circ_Put:
	cli				;
	push	TEMP			;Save the data
	ld	TEMP,Y			;Get the size
	cp	TEMP2,TEMP		;Compare to the length
	breq	Circ_Put_Done_Full	;If full, then nothing to do.
	inc	TEMP			;increment that, since we stored something
	st	Y,TEMP			;and put it away

	ldd	TEMP,Y+2		;Get the tail pointer
	inc	TEMP			;Bump it
	cp	TEMP,TEMP2		;Did it wrap? If Temp > TEMP2, then yes
	brlo	Circ_Put_Go		;If less than
	breq	Circ_Put_Go		;Or equal, it's ok
	ldi	TEMP,1			;An incremented zero!

Circ_Put_Go:				;It's a go, let's do it.
	std	Y+2,TEMP		;Still pointing to the size

	add	YL,TEMP			;Add the tail pointer to the base
	adc	YH,ZERO			;ZERO is always zero, but carry isn't
	pop	TEMP			;
	std	Y+2,TEMP		;The offset gives us data
	sei				;
	sec				;Signal Success
	ret				;

Circ_Put_Done_Full:			;Nothing stored!!
	pop	TEMP			;
	sei				;
	clc				;
	ret				;
;
;******************************************************************
;
;Stores a char in TEMP to the pointed circular buffer.
;Needs buffer length in TEMP2 and buffer pointer in Y
;
Circ_Get:
	;Temp is junk, so we don't push it.
	cli				;No ints!
	ld	TEMP,Y			;Get the size (now pointing at the head)
	and	TEMP,TEMP		;
	breq	Circ_Get_Done_Empty	;If full, then nothing to do.
	dec	TEMP			;
	st	Y,TEMP			;

Circ_Get_Go:
	ldd	TEMP,Y+1		;Get the head pointer
	inc	TEMP			;Bump it
	cp	TEMP,TEMP2		;Will it wrap?
	brlo	Circ_Get_Go_A		;If less than, then no
	breq	Circ_Get_Go_A		;If equal then no.
	
	ldi	TEMP,1			;an incremented zero

Circ_Get_Go_A:
	std	Y+1,TEMP		;Put the head pointer back
	add	YL,TEMP			;
	adc	YH,ZERO			;Now pointing at the data
	ldd	TEMP,Y+2		;Get the data
	sei				;
	sec				;Signal Success
	ret				;

Circ_Get_Done_Empty:
	sei				;
	clc				;
	ret				;

;
;******************************************************************
;
;Scans for a char in the buffer, returns it's position in TEMP if found.
;Starts at the beginning, and scans to the end.
;NOTE: If other tasks are modifying the buffer, the results may not represent
;reality for long. You may want to disable buffer modifiers while you scan.
;
;Carry set for found, carry cleared for not found
;Y points to the buffer
;Temp2 has buffer length
;Temp has the char to match
;TEMP3 trashed.
;
Circ_Scan:
	ldi	TEMP3,0			;Set to the beginning of the buffer
	mov	LOOP,TEMP3		;Loop carries where we are looking
	mov	TEMP3,TEMP		;

Circ_Scan_Loop:
	mov	TEMP,LOOP		;TEMP now says where to look
					;TEMP2 has the length of the buffer
	rcall	Circ_Read		;What's at the LOOP'th position of the buffer?
					;Output in TEMP
	cp	TEMP,TEMP3		;compare the stored copy
	breq	Circ_Scan_Done		;Equal, then loop says where.
	inc	LOOP			;Otherwise keep searching
	ld	TEMP2,Y			;
	cp	LOOP,TEMP2		;Unless we've gone past the end
	brne	Circ_Scan_Loop		;

Circ_Scan_Nothing:
	clc				;Signal Failure
	ret				;

Circ_Scan_Done:
	sec				;Signal Success
	mov	TEMP,LOOP		;And say where
	ret				;
;
;******************************************************************
;
;Reads the char at a position within the buffer, without eating it.
;Position input in TEMP 
;Char returned in TEMP
;Y points to the buffer
;Buffer length in TEMP2
;
Circ_Read:
	push	YL			;So the Y is preserved
	push	YH			;
	push	TEMP2			;We use TEMP2

	ld	TEMP2,Y			;How many are stored
	and	TEMP2,TEMP2		;Is it zero?
	breq	Circ_Read_Empty		;If so, signal and return

	ldd	TEMP2,Y+1		;Get the head pointer
	add	TEMP2,TEMP		;
	pop	TEMP			;Get the length back
	cp	TEMP,TEMP2		;
	brge	Circ_Read_It		;
	sub	TEMP,TEMP2		;Handle Wrap

Circ_Read_It:
	add	YL,TEMP2		;Point at the data
	adc	YH,ZERO			;
	ldd	TEMP,Y+3		;Get it, remembering the SHT header!
	rjmp	Circ_Read_Done		;And we're outta here

Circ_Read_Empty:
	clc				;Signal failure
	pop	TEMP2			;Balance the stack

Circ_Read_Done:
	pop	YH			;
	pop	YL			;
	ret				;Bye

;
;******************************************************************
;
;