
; File Name		:'LCD.asm"
; Title			:LCD routines
; Date			:
; Version		:
; Support telephone	:765 287 1987  David B. VanHorn
; Support fax		:765 287 1989
; Support Email		:dvanhorn@cedar.net
; Target MCU		:AT90S8515
;
; DESCRIPTION
; Generic single or dual line display, 8-40 chars, in 4 bit mode
;
;***************************************************************************;
;	M O D I F I C A T I O N   H I S T O R Y 
;
;
;       rev.      date    who   why
;	----	--------  ---	------------------------------------------
;	0.01	98.11.27  dvh	Creation, working
;	0.02	99.09.10  dvh	Made to work with new buffer managers,
;				and in place alongside the VFD module,
;				provided you relocate the hardware.
;				Small revisions to the scroll routines.
;				Converted to work on a single port, but
;				still easily capable of splitting control
;				and data to different ports. Tricky LCD
;				power hookup for fast testing.
;				Main routines working, but only scroll all
;				left in the scroll routines is working.
;				Thinking about exporting the scroll routines
;				to memory.asm as they are really generic 
;				buffer manipulations.
;				All routines tested ok
;
;
;***************************************************************************
;Pin
;number	Symbol	I/O	Function
;    1	Vss	-	Power supply (GND)
;    2	Vcc	-	Power supply (+5V)
;    3	Vee	-	Contrast adjust
;    4	RS	I	0 = Instruction input
;                        1 = Data input
;    5	R/W	I	0 = Write to LCD module
;			1 = Read from LCD module
;    6	E	I	Enable signal
;    7	DB0	I/O	Data bus line 0 (LSB)
;    8	DB1	I/O	Data bus line 1
;    9	DB2	I/O	Data bus line 2
;    10	DB3	I/O	Data bus line 3
;    11	DB4	I/O	Data bus line 4
;    12	DB5	I/O	Data bus line 5
;    13	DB6	I/O	Data bus line 6
;    14	DB7	I/O	Data bus line 7 (MSB)
;
;***************************************************************
;Display Test Cable Wiring to port A
;Pin
;number	Symbol	Cable	Function
;    1	Vss	Pin9	Power supply (GND)
;    2	Vcc	Pin5	PA4 Tricky, powering the display from an output bit!
;    3	Vee	Pin10	GND	
;    4	RS	Pin8	PA7
;    5	R/W	Pin7	PA6
;    6	E	Pin6	PA5
;    7	DB0	Open
;    8	DB1	Open
;    9	DB2	Open
;    10	DB3	Open
;    11	DB4	Pin1	PA0 Low bit
;    12	DB5	Pin2	PA1
;    13	DB6	Pin3	PA2
;    14	DB7	Pin4	PA3 High bit
;*****************************************************************
;HD44780 instruction set
;
;Instruction Code 		Description
;
;	R R D D D D D D D D
;	S W 7 6 5 4 3 2 1 0
;Clear	0 0 0 0 0 0 0 0 0 1  Clears display and returns cursor to the home position (address 0) 1.64mS
;
;Home   0 0 0 0 0 0 0 0 1 X  Returns cursor to home position (address 0). 
;			     Also returns display being shifted the original position. 
;			     DDRAM contents remains unchanged. 1.64mS
;Entry mode set
;	0 0 0 0 0 0 0 1 I/D S	Sets cursor move direction (I/D), specifies to shift the display (S).
;				These operations are performed during data read/write. 40uS
;Display On/Off control
;	0 0 0 0 0 0 1 D C B	Sets On/Off of all display (D), cursor On/Off (C) and blink of cursor 
;				position character (B). 40uS
;Cursor/display shift
;	0 0 0 0 0 1 S/L R/L X X	Sets cursor-move or display-shift (S/C), shift direction (R/L).
;				DDRAM contents remains unchanged. 40uS
;Function set
;	0 0 0 0 1 DL N F X X	Sets interface data length (DL), number of display line (N) 
;				and character font(F). 40uS
;Set CGRAM address
;	0 0 0 1 CGRAM address	Sets the CGRAM address. 
;				CGRAM data is sent and received after this setting. 40uS
;Set DDRAM address
;	0 0 1 DDRAM address	Sets the DDRAM address.
;				DDRAM data is sent and received after this setting. 40uS                                                                         40uS
;Read busy-flag and address counter
;	0 1 BF DDRAM address	Reads Busy-flag (BF) indicating internal operation
;				is being performed and reads address counter contents. 0uS
;Write to CGRAM or DDRAM
;	1 0 write data		Writes data to CGRAM or DDRAM. 40uS
;Read from CGRAM or DDRAM
;	1 1 read data 		Reads data from CGRAM or DDRAM. 40uS
;
;********************************************************************************
;
;	Bit name
;	I/D	0 = Decrement cursor position
;		1 = Increment cursor position
;
;	S	0 = No display shift
;		1 = Display shift
;
;	D	0 = Display off
;		1 = Display on
;
;	C	0 = Cursor off
;		1 = Cursor on
;
;	B	0 = Cursor blink off
;		1 = Cursor blink on
;
;	S/C	0 = Move cursor
;		1 = Shift display
;
;	R/L	0 = Shift left
;		1 = Shift right
;
;	DL	0 = 4-bit interface
;		1 = 8-bit interface
;
;	N	0 = 1/8 or 1/11 Duty (1 line)
;		1 = 1/16 Duty (2 lines)
;
;	F	0 = 5x7 dots
;		1 = 5x10 dots
;
;	BF	0 = Can accept instruction
;		1 = Internal operation in progress
;
;********************************************************
;
;Shown after reset (with N=0).
;
;Disp	Positions	DDRAM addresses
; 1*8	00..07		00h..07h
; 1*16	00..15		00h..0Fh
; 1*20	00..19		00h..13h
; 1*24	00..23		00h..17h
; 1*32	00..31		00h..1Fh
; 1*40	00..39		00h..27h
;
;*************************************************
;
;Shown after reset (with N=1).
;
;Disp	Positions	DDRAM addresses
;2*16	00..15		00h..0Fh + 40h..4Fh
;2*20	00..19		00h..13h + 40h..53h
;2*24	00..23		00h..17h + 40h..57h
;2*32	00..31		00h..1Fh + 40h..5Fh
;2*40	00..39		00h..27h + 40h..67h
;*****************************************************
;
;Shown after reset (with N=1).
;                                                                         
;Disp	Positions	DDRAM addresses
;4*16	00..15		00h..0Fh + 40h..4Fh + 14h..23h + 54h..63h
;4*20	00..19		00h..13h + 40h..53h + 14h..27h + 54h..67h
;
;************************************************************
;
;THINGS YOU NEED TO ALLOCATE 
;
;You have to define things, before you can use them.
;Below, in the CSEG (Code Segment) I define some constants.
;Then, in the DSEG (Data segment) I use them to allocate some memory.
;If you reverse this order, the assembler may not pick up an error,
;But you will experience "non-deterministic execution".
;
;
.CSEG
.equ	LCD_Lines=2
.equ	LCD_LEN=16
.equ	LCD_OUT_Size=LCD_Lines * LCD_Len
.equ	LCD_Scroll_Speed = 200	;ms per action, used with LCD_Timer

.DSEG
LCD_Timer:	.byte	1
LCD_OUT_BUF:	.byte	LCD_OUT_Size+2	;Size defined above
LCD_BITFLAGS:	.byte	1	;7=
				;6=
				;5=
				;4=
				;3=
				;2=
				;1=
				;0=
.CSEG

;*************************************************************
;
;HARDWARE YOU NEED TO SET UP
;
;Hardware pins assigned here to logical names.
;
;Warning: Port D shares with the Uart.
;
.equ	LCD_DATA=PORTB	;LCD data lines interface
			;Only D7-D4 used. D3-0 are open
.equ	LCD_D_DDR=DDRB	;So we can shift the bits from I to O
.equ	LCD_C_DDR=DDRB	;

.equ	LCD_INPUT=PINB	;LCD Input Pins, as above
.equ	LCD_CTRL=PORTB	;LCD control lines interface
.equ	LCD_D0=0	;Defining where and how the data bits
.equ	LCD_D1=1	;are connected
.equ	LCD_D2=2	;
.equ	LCD_D3=3	;
.equ	LCD_PWR=4	;ONLY use for my tricky plug
.equ	LCD_E=5		;LCD Enable control line
.equ	LCD_RW=6	;LCD Read/Write control line
.equ	LCD_RS=7	;LCD Register-Select control line
;
;This is an example only, you can vary these settings, but this 
;is what I used.
;
;************************************************************
;
;Call LCD_Init after powerup, to clear the display and get it
;ready to talk.
;
;First, call Say_LCD_Test, which will put the display image into
;the buffer in RAM. Then call Check_LCD_Display, which will cause
;the actual display to be updated.
;
;Make more prompt routines as needed.
;
;Check_LCD_Display should be the lowest priority task, run whenever
;you aren't doing anything special, or run it if it's important
;that the display get updated immediately.
;
;The scroll routines operate on the display buffer, and set the
;display dirty flag so that Check_LCD_Display will update the screen
;
;************************************************************
;Prompt routines
;************************************************************
;                        12345678901234567890123456789012
Test_MSG:	.db	"ABCDEFGHIJKLMNOPQRSTUVWXYZ123456",0
Say_LCD_Test:
	ldi	ZL,low(Test_MSG*2)		;Make the Z reg point at the table
	ldi	ZH,high(Test_MSG*2)		;preparing for the LPM instruction
	rcall	LCD_STR_Nopad			;String to buf, buf to LCD
	ret
;
;***************************************************************************
;Scroll the display, only if it's time to do so.
;***************************************************************************
;
LCD_Scroll:
	lds	Temp,LCD_Timer		;Get the system timer (dec'd by T0)
	and	TEMP,TEMP		;Check it
	brne	LCD_Scroll_Done		;If non-zero, don't bother
	ldi	TEMP,LCD_Scroll_Speed	;Otherwise reload the timer first
	sts	LCD_Timer,TEMP		;so the task time dosen't bother
					;the planned repition rate

	;Now perform scrolling as desired

	;rcall	Scroll_All_LCD_Left	;Working
	;rcall Scroll_Top_LCD_Left	;Working
	rcall Scroll_Bot_LCD_Left	;Working
	;rcall	Scroll_All_LCD_Right	;Working
	rcall Scroll_Top_LCD_Right	;Working
	;rcall Scroll_Bot_LCD_Right	;Working

LCD_Scroll_Done:
	ret
;
;***************************************************************************
;Update the display, only if needed
;***************************************************************************
;
Check_LCD_Display:
	lds	TEMP,LCD_Bitflags
	andi	TEMP,$40	;Remove all but the "Dirty Bit"
	breq	CK_LCD_Disp_Done;If zero, then just exit
	rcall	LCD_Spew	;Otherwise, update the display

	lds	TEMP,LCD_Bitflags
	andi	TEMP,$BF	;Reset "dirty bit"
	sts	LCD_BitFlags,TEMP

CK_LCD_Disp_Done:
	ret			;
;
;****************************************************************
;
;These routines roll the entire display memory from left to right, or right to left.
;
Scroll_All_LCD_Left:
	push	TEMP			;
	push	TEMP2			;
	push	LOOP			;

	ldi	ZH,high(LCD_OUT_BUF)	;Point at the beginning
	ldi	ZL,low(LCD_OUT_BUF)	;
	ld	TEMP,Z+			;Current size
	ld	TEMP,Z+			;Temp has length
	dec	TEMP			;-1
	mov	LOOP,TEMP		;	
	mov	YL,ZL			;
	mov	YH,ZH			;
	ld	TEMP2,Z+		;Pick up the first one

	;At this point, Z points to buffer+1, and Y points to buffer.
	;Temp2 contains the first char, which will be over-written by
	;the first shift.
	rjmp	Scroll_LCD_Left_Loop	;

Scroll_Top_LCD_Left:
	push	TEMP			;
	push	TEMP2			;
	push	LOOP			;

	ldi	ZH,high(LCD_OUT_BUF)	;Point at the beginning
	ldi	ZL,low(LCD_OUT_BUF)	;
	ld	TEMP,Z+			;Get size
	ld	TEMP,Z+			;Get length
	mov	YL,ZL			;Make Y point at the first byte
	mov	YH,ZH			;
	clc				;Half the length
	ror	TEMP			;
	dec	TEMP			;One less
	mov	LOOP,TEMP		;
	ld	TEMP2,Z+		;Get the end byte, and bump Z

	;At this point, Z points to buffer+1, and Y points to buffer.
	;Temp2 contains the first char, which will be over-written by
	;the first shift.
	rjmp	Scroll_LCD_Left_Loop	;

Scroll_Bot_LCD_Left:
	push	TEMP			;
	push	TEMP2			;
	push	LOOP			;

	ldi	ZH,high(LCD_OUT_BUF)	;Point at the beginning
	ldi	ZL,low(LCD_OUT_BUF)	;
	ld	TEMP,Z+			;Get size
	ld	TEMP,Z+			;Get length
	clc				;
	ror	TEMP			;Half that
	dec	TEMP			;One less
	mov	LOOP,TEMP		;
	
	clc
	inc	TEMP			;
	add	ZL,TEMP			;Add half a buffer
	adc	ZH,ZERO			;
	mov	YL,ZL			;Make Y point here
	mov	YH,ZH			;

	ld	TEMP2,Z+		;Pick up the first one
	rjmp	Scroll_LCD_Left_Loop	;
;
;The Y and Z pointers are set, and TEMP2 contains the end char, TEMP has the 
;number of chars to scroll. All we do here is pick them up, and set them down.
;
Scroll_LCD_Left_Loop:

	ld	TEMP,Z+			;Pick up the second char and then point at 
					;the third
	st	Y+,TEMP			;Store second at first, and then point at 
					;the second.	
	dec	Loop			;One less to shift.
	brne	Scroll_LCD_Left_Loop	;
	
	st	Y,TEMP2			;Put the first char in the last cell
	lds	TEMP,LCD_Bitflags	;
	ori	TEMP,$40		;Flag the display dirty
	sts	LCD_Bitflags,TEMP	;

	pop	LOOP			;
	pop	TEMP2			;
	pop	TEMP			;
	ret



Scroll_All_LCD_Right:
	push	TEMP			;
	push	TEMP2			;
	push	LOOP			;

	ldi	ZH,high(LCD_OUT_BUF)	;Point at the beginning
	ldi	ZL,low(LCD_OUT_BUF)	;
	ld	TEMP,Z+			;Get size
	ld	TEMP,Z+			;Get length
	mov	LOOP,TEMP		;
	dec	TEMP			;Fix pointer in by one

	clc				;Point at the end of the buffer
	add	ZL,TEMP			;
	adc	ZH,ZERO			;
	mov	YL,ZL			;Y points here also
	mov	YH,ZH			;

	ld	TEMP2,Z			;Pick up the first one
	ld	R0,-Z			;Post dec
	rjmp	Scroll_LCD_Right_Loop	;

Scroll_Top_LCD_Right:
	push	TEMP			;
	push	TEMP2			;
	push	LOOP			;

	ldi	ZH,high(LCD_OUT_BUF)	;Point at the beginning
	ldi	ZL,low(LCD_OUT_BUF)	;
	ld	TEMP,Z+			;
	ld	TEMP,Z+			;
	clc				;
	ror	TEMP			;
	mov	LOOP,TEMP		;
	dec	TEMP			;Fix pointer in by one
	clc				;
	add	ZL,TEMP			;
	adc	ZH,ZERO			;
	mov	YL,ZL			;
	mov	YH,ZH			;

	ld	TEMP2,Z			;Pick up the first one
	ld	R0,-Z			;Toss
	rjmp	Scroll_LCD_Right_Loop	;

Scroll_Bot_LCD_Right:
	push	TEMP			;
	push	TEMP2			;
	push	LOOP			;
	
	ldi	ZH,high(LCD_OUT_BUF)	;Point at the beginning
	ldi	ZL,low(LCD_OUT_BUF)	;
	ld	TEMP,Z+			;Skip size
	ld	TEMP,Z			;Get length
	clc				;
	ror	TEMP			;Half that
	mov	LOOP,TEMP		;Save how many to do
	
	ld	TEMP,Z+			;Get the original length back
	dec	TEMP			;Fix pointer in by one
	clc				;
	add	ZL,TEMP			;Point at the 
	adc	ZH,ZERO			;
	mov	YL,ZL			;
	mov	YH,ZH			;


	ld	TEMP2,Z				;Pick up the first one
	ld	R0,-Z				;Toss
	rjmp	Scroll_LCD_Right_Loop		;
;
;Same as Scroll_Left_Loop, but note the pre-dec rather than post inc.
;
Scroll_LCD_Right_Loop:

	ld	TEMP,Z			;Pick it up here
	ld	R0,-Z			;This just does a post dec on Z
	st	Y,TEMP			;Put it down there
	ld	R0,-Y			;And a post dec on Y
	dec	Loop			;One less to mess with
	brne	Scroll_LCD_Right_Loop	;If not done, do more
	
	ld	TEMP,Y+			;Fix the end pointer
	st	Y,TEMP2			;Put the first char in the last cell
	lds	TEMP,LCD_Bitflags	;
	ori	TEMP,$40		;Flag the display dirty
	sts	LCD_Bitflags,TEMP	;

	pop	LOOP			;
	pop	TEMP2			;
	pop	TEMP			;

	ret
;
;****************************************************************
;Called with Z pointing to a string in rom.
;Seeks and displays forward, until we find a null.
;Shoves that text into the LCD_OUT_BUF
;****************************************************************	
;
LCD_STR_NOPAD:

	ldi	YH,high(LCD_OUT_BUF)	;
	ldi	YL,low(LCD_OUT_BUF)	;
	rcall	String_Out		;MEMORY.ASM
	lds	TEMP,LCD_Bitflags	;
	ori	TEMP,$40		;Flag the display dirty
	sts	LCD_Bitflags,TEMP	;
	ret

LCD_STR_PAD:
	rcall	LCD_STR_NOPAD	;This moves the string
	;Point at the beginning of the LCD buffer
	ldi	YH,high(LCD_OUT_BUF)	;
	ldi	YL,low(LCD_OUT_BUF)	;
	ld	TEMP2,Y+		;Get length
	ld	TEMP3,Y+		;Get size
	cp	TEMP3,TEMP2		;Is it full?
	breq	LCD_Fill_Done		;If so, all done

	ldi	TEMP,$20		;A Space
	ldi	YH,high(LCD_OUT_BUF)	;Point
	ldi	YL,low(LCD_OUT_BUF)	;
	rcall	Store_Buffer		;Stuff it
	rjmp	LCD_STR_Pad		;And maybe do it again

LCD_Fill_Done:
	lds	TEMP,LCD_Bitflags	;
	ori	TEMP,$40		;Flag the display dirty
	sts	LCD_Bitflags,TEMP	;
	ret
;
;**************************************************************
;This routine moves the data in LCD_OUT_BUF to the display.
;Called only from IDLE, this updates the display to be current 
;with the contents of the lcd buffer. 
;**************************************************************
;
LCD_SPEW:
	push	TEMP			;
	push	TEMP2			;
	push	LOOP			;

	rcall	LCD_Clear		;Clear and home, all delays

	;Point at the beginning of the LCD buffer
	ldi	ZH,high(LCD_OUT_BUF)	;
	ldi	ZL,low(LCD_OUT_BUF)	;
	ld	TEMP2,Z+		;
	ld	TEMP2,Z+		;Get the len, pointing at data
	mov	LOOP,TEMP2		;This says how many to send
	clc				;
	ror	TEMP2			;So we know where half is
	rjmp	LCD_Spew_L2		;Skip past the test

LCD_SPEW_Loop:
	cp	LOOP,TEMP2		;Are we pointing to the second line?
	brne	LCD_SPEW_L2		;
	rcall	Bump_Pointer		;We only do this once, in the 
					;case of "equals".

LCD_Spew_L2:
	ld	TEMP,Z+			;Get a char, and inc Z
	rcall	LCD_PUT_CHAR		;
	dec	LOOP			;One less to send
	brne	LCD_SPEW_Loop		;If not zero, keep dumping

LCD_SPEW_Done:
	lds	TEMP,LCD_Bitflags	;
	andi	TEMP,$BF		;Flag the display clean
	sts	LCD_Bitflags,TEMP	;
	pop	LOOP			;
	pop	TEMP2			;
	pop	TEMP			;

	ret				;
;
;*************************************************************
;This just puts the pointer onto the start of the 2nd line
;*************************************************************
;
Bump_Pointer:
	ldi	TEMP,$C0	;Set the display pointer to 40
	rcall	LCD_PUT_CMD	;
	ret
;
;*************************************************************
;The main powerup init routine. Only called after powerup
;*************************************************************
;
LCD_INIT:	

	sbi	LCD_CTRL,LCD_PWR;Bring bit 4 high for power!
				;NOTE! This is unique to my test
				;setup that allows me to plug the 
				;LCD into PORT(X) to easily verify
				;the code. Normally this is a waste
				;of a nice I/O pin, and you wouldn't
				;do this

	cbi	LCD_CTRL,LCD_RS	;Normal ops
	cbi	LCD_CTRL,LCD_RW	;Read
	cbi	LCD_CTRL,LCD_E	;Not strobing
	

	sbi	LCD_C_DDR,LCD_RS;Make all the command pins outputs.
	sbi	LCD_C_DDR,LCD_RW;	
	sbi	LCD_C_DDR,LCD_E	;
	sbi	LCD_C_DDR,LCD_PWR;

	sbi	LCD_D_DDR,LCD_D0;Make the data pins outputs
	sbi	LCD_D_DDR,LCD_D1;
	sbi	LCD_D_DDR,LCD_D2;
	sbi	LCD_D_DDR,LCD_D3;

	
	; Busy-flag is not yet valid for 15mS after powerup
	ldi	TEMP,16		;>15 by spec
	rcall	MS_Delay	;So we wait 16ms
	ldi	TEMP,$30	;8-bit-interface, 2-lines, 5x7 font
	rcall   LCD_Slam_CMD	;Don't look for busy, just send.

	ldi	TEMP,5		;>4.1 by spec
	rcall	MS_Delay	;
	ldi	TEMP,$30	;8-bit-interface, 2-lines, 5x7 font
	rcall   LCD_Slam_CMD	;

	ldi	TEMP,1		;>100uS
	rcall	MS_Delay	;
	ldi	TEMP,$30	;8-bit-interface, 2-lines, 5x7 font
	rcall   LCD_Slam_CMD	;

	ldi	TEMP,1		;No specified delay
	rcall	MS_Delay	;
	ldi	TEMP,$20	;4-bit-interface, 2-lines, 5x7 font
	rcall   LCD_Slam_CMD	;

	ldi	TEMP,1		;No specified delay
	rcall	MS_Delay	;
	ldi	TEMP,$0C	;disp on, curs.off, no-blink
	rcall	LCD_Slam_CMD	;

	ldi	TEMP,1		;No specified delay
	rcall	MS_Delay	;
	ldi	TEMP,$01	;Clear
	rcall	LCD_Slam_CMD	;

	ldi	TEMP,1		;No specified delay
	rcall	MS_Delay	;
	ldi	TEMP,$06	;Entry set
	rcall	LCD_Slam_CMD	;

	cbi	LCD_CTRL,LCD_RS	;
	sbi	LCD_CTRL,LCD_RW	;
	cbi	LCD_CTRL,LCD_E	;
	sbi	LCD_CTRL,4	;Keep power on

	ret
;
;************************************************************
;Detect wether the display is busy or not.
;************************************************************
;
LCD_BUSY:
	push	TEMP		;There's a char for the LCD in here

LCD_Busy_Loop:

	cbi	LCD_Data,3	;de-Activate pullup
	;This is done so that the system won't hang if there's 
	;no display connected.

	cbi	LCD_D_DDR,3	;Make bit 3 an input

        cbi	LCD_CTRL,LCD_RS	;Set LCD for command mode
	sbi	LCD_CTRL,LCD_RW	;Set read mode
	cbi	LCD_CTRL,LCD_E 	;LCD E-line low
	rjmp	LCD_Busy_Loop_A	;Go to the first half

LCD_Busy_Loop_B:
	cbi	LCD_CTRL,LCD_E 	;LCD E-line low (end of high nybble)
	nop			;
	sbi	LCD_CTRL,LCD_E 	;LCD E-line High (Read low nybble)
	cbi	LCD_CTRL,LCD_E 	;LCD E-line Low  dummy op


LCD_Busy_Loop_A:
	rcall	Timed_Smack	;Since we could loop here a while...

	sbi	LCD_CTRL,LCD_E 	;LCD E-line High
	nop			;Min 320nS before read
	nop			;

	sbic	LCD_Input,3	;If bit 3 is set, then we're done
	rjmp	LCD_Busy_Loop_B	;


	;Finish off this cycle
	cbi	LCD_CTRL,LCD_E 	;LCD E-line low
	nop			;
	sbi	LCD_CTRL,LCD_E 	;LCD E-line High
	cbi	LCD_CTRL,LCD_E 	;LCD E-line Low
	sbi	LCD_D_DDR,3	;Make the data pin an output again

	;make the LCD Data port an output again, TEMP2 has RAM adrs
	sbi	LCD_D_DDR,3	;

	pop	TEMP		;Put back the char to the LCD
	ret
;
;*************************************************************
;Clear and home the display
;*************************************************************
;
LCD_CLEAR:
        ldi	TEMP,$01	;The "Clear Display" command
        rcall	LCD_Slam_CMD	;
        ldi	TEMP,$02	;The "Home" command
        rcall	LCD_PUT_CMD	;Wait till not busy, then send.
	ret
;
;**************************************************************
;Sends character to LCD
;Required character must be in TEMP
;**************************************************************
;
LCD_PUT_CHAR:
	rcall	LCD_BUSY	;Wait for LCD to be ready (Busy saves TEMP)
	sbi	LCD_CTRL,LCD_RS	;Set LCD in data mode
	rjmp	LCD_Talk	;
;
;**************************************************************
;Sends command to LCD
;Required command must be in TEMP
;**************************************************************
;
LCD_PUT_CMD:
	rcall	LCD_BUSY	;Wait for LCD to be ready (Busy saves TEMP)

LCD_Slam_CMD: ;Just like put, but no test for busy
	cbi	LCD_CTRL,LCD_RS	;Set LCD in command mode
	;rjmp	LCD_Talk	;Fallthru
;
;**************************************************************
;The basic engine that talks to the LCD.
;Command or Data mode were set up above, in PUT_CMD, PUT_CHAR.
;SLAM_CMD exists only to serve for sending commands during the
;time that the busy flag can't be tested.
;**************************************************************
;
LCD_Talk:
	
	cli	;DEBUG
	
	cbi	LCD_CTRL,LCD_RW	;Make RW low
	nop			;125ns each, I need 140nS by the book
	nop			;
	cbi	LCD_CTRL,LCD_E	;and E is low

 	push	TEMP		;Save a copy
	andi	TEMP,$F0	;Take the high nybble first
	swap	TEMP		;
	rcall	LCD_Talk_Dbits	;Takes E low, then high again

	pop	TEMP		;Retrieve copy
	andi	TEMP,$0F	;Now do the low nybble
	rcall	LCD_Talk_Dbits	;Takes E low, then high again

LCD_Talk_Fin:

	sei	;DEBUG

	ret		

	;Just for fun, I set the bits individually, meaning they
	;could really be anywhere, even on different ports if you 
	;change the port labels appropriately.

LCD_Talk_Dbits:
	sbi	LCD_D_DDR,LCD_D0;Make the data pins outputs
	sbi	LCD_D_DDR,LCD_D1;
	sbi	LCD_D_DDR,LCD_D2;
	sbi	LCD_D_DDR,LCD_D3;
	
	ror	TEMP		;
	brcs	LCD_TAlk_D0H	;
	cbi	LCD_DATA,LCD_D0	;
	rjmp	LCD_Talk_D1	;	
LCD_Talk_D0H:
	sbi	LCD_DATA,LCD_D0	;

LCD_Talk_D1:
	ror	TEMP		;
	brcs	LCD_TAlk_D1H	;
	cbi	LCD_DATA,LCD_D1	;
	rjmp	LCD_Talk_D2	;	
LCD_Talk_D1H:
	sbi	LCD_DATA,LCD_D1	;

LCD_Talk_D2:
	ror	TEMP		;
	brcs	LCD_TAlk_D2H	;	
	cbi	LCD_DATA,LCD_D2	;
	rjmp	LCD_Talk_D3	;	
LCD_Talk_D2H:
	sbi	LCD_DATA,LCD_D2	;

LCD_Talk_D3:
	ror	TEMP		;
	brcs	LCD_TAlk_D3H	;
	cbi	LCD_DATA,LCD_D3	;
	rjmp	LCD_Talk_DFin	;	
LCD_Talk_D3H:
	sbi	LCD_DATA,LCD_D3	;


LCD_Talk_DFin:
	sbi	LCD_CTRL,LCD_E	;Toggle E High
	nop			;
	cbi	LCD_CTRL,LCD_E	;Back to Low
	ret			;
;
;************************************************************
