; 
; File Name		:'ISR.asm"
; Title			:8515 Interrupt Handlers
; Date			:
; Version		:
; Support telephone	:765 287 1987  David B. VanHorn
; Support fax		:765 287 1989
; Support Email		:dvanhorn@cedar.net
; Target MCU		:AT90S8515
;
; DESCRIPTION
; All interrupt handlers reside here
;
;***************************************************************************;
;	M O D I F I C A T I O N   H I S T O R Y 
;
;
;       rev.      date    who   why
;	----	--------  ---	------------------------------------------
;	0.01	98.07.29  dvh	Creation
;	0.02	98.08.20  dvh	Added Timer 1 int for servo control, a little cleanup
;				and better documentation elsewhere
;	0.03	98.08.21  dvh	Added support in TIM0_OVF for a FRAME_DELAY
;				variable used in SERVO.ASM. 
;				Converted Frame_Delay to SRAM from register
;	0.04	98.08.23  dvh	Regularized to XL,XH notation instead of R26.R27 etc
;	0.05	99.09.09  dvh	Fixed morse handler to work properly.
;	0.06	99.09.13  dvh	Modified to work with new circular buffer
;	0.07	99.09.17  dvh	Many updates to UDRIE, RXC and T0 to accomodate
;				full duplex serial comms, with hardware and 
;				xon/xoff handshaking. Hardware working and tested
;				xon not tested yet.
;	0.08	99.10.20  dvh	Added EEPROM_Timer in T0 ISR
;***************************************************************************
;Fixed jump vectors
;
;$000
	rjmp	INIT_Machine	;Restart vector, points to the beginning of the code. 
;$001
	rjmp	EXT_INT0	;IRQ 0
;$002
	rjmp	EXT_INT1	;IRQ 1 
;$003	
	rjmp	TIM1_CAPT	;Timer 1 capture 
;$004
	rjmp	TIM1_COMPA	;Timer 1 compare A
;$005
	rjmp	TIM1_COMPB	;Timer 1 compare B
;$006
	rjmp	TIM1_OVF	;Timer 1 overflow
;$007	
	rjmp	TIM0_OVF	;Timer 0 overflow
;$008	
	rjmp	SPI_Handler	;Serial transfer complete
;$009
	rjmp	UART_RXC	;Uart RX complete
;$00A
	rjmp	UART_DRE	;Uart data register empty
;$00B
	rjmp	UART_TXC	;Uart TX complete
;$00C	
	rjmp	ANA_COMP	;Analog comparator
;
;********
;
;All my ISRs use dedicated registers TTEMP and TTEMP2 to maximize speed. 
;You could push TEMP, and pop it back if you really need registers.
;
;Each int vector has, at minimum, code that turns off that int. 
;This is a "belts and suspenders" method that protects me from
;any ints I forgot to turn off, or accidentally enabled.
;
;If you decide to use one of the disabled ints, enable it in the
;code when appropriate, and replace these "turn-off" handlers
;with whatever it is you want done.
;
;Each ISR saves SREG to TTEMP2, which is dedicated to this purpose.
;I could have pushed and popped it, but push and pop take two cycles,
;and in and out only take one. If you get squeezed for registers, then
;you can always do the pushpop dance. It's an extra 500nS at 8MHz
;
;MAKE ABSOLUTELY SURE YOU DO NOT USE "TEMP" IN ANY ISR! (unless you push and pop it!)
;
;
;********
;
;Only used with the ISR debug statements. Set to any free pin
;
.equ	Pip_Port=PORTA	;
.equ	Pip_DDR=DDRA	;
.equ	Pip_Bit=1	;
;
;********
;
;Unused at this point. 
;
;
EXT_INT0:	
	in	TTEMP2,SREG	;Saves the status register
	
	;Since I use dedicated TTEMP registers for the interrupts, this is
	;all I need to do before acting on the int.  At the end of the isr,
	;SREG is restored, and we exit. 

	;The following code is a simple diagnostic, that can help you debug
	;interrupts. As shown, you would uncomment the debug code, and get 
	;a pulse on a pin during an interrupt. The width of the pulse shows 
	;how much time the ISR takes (plus 450nS for the debug instructions)
	;You can also put multiple SBI/CBI instructions at the beginning of
	;the ISRs, so that you can tell which is firing, if you have multiple
	;ISRs enabled to debug.

	;DEBUG
	;sbi	Pip_DDR,Pip_Bit	;Make it an output
	;sbi	Pip_Port,Pip_Bit;


	;Since this int isn't used in the app, I just leave it with the default
	;handler, which turns off this int. I should never have come here in the
	;first place, since this int is supposed to be off. This is VERY cheap
	;protection.  Some people code only the ISRs that they want. This can 
	;have two bad effects. One, if all you have in the table is a RETI, then
	;a wild int can eat up a ton of CPU time.
	;Two, if you don't have at least the RETI handlers for each int, then 
	;the wild int will cause the execution of the next int handler in the 
	;table, which could be a debug nightmare.

INT0_OFF:
	in	TTEMP,GIMSK	;If done, then turn off the int!
	andi	TTEMP,$BF	;
	out	GIMSK,TTEMP	;

INT0_EXIT:

	;DEBUG
	;cbi	Pip_Port,Pip_Bit;

	out	SREG,TTEMP2	;restore the status register
	reti
;
;********
;
;
EXT_INT1:
	in	TTEMP2,SREG	;Saves the status register

	;DEBUG
	;sbi	Pip_DDR,Pip_Bit	;Make it an output
	;sbi	Pip_Port,Pip_Bit;

INT1_OFF:
	in	TTEMP,GIMSK	;Get the int enables
	andi	TTEMP,$7F	;Mask off INT1
	out	GIMSK,TTEMP	;Send it
	ldi	TTEMP,$80	;
	out	GIFR,TTEMP	;Manual clear
	
INT1_EXIT:

	;DEBUG
	;cbi	Pip_Port,Pip_Bit;

	out	SREG,TTEMP2	;restore the status register
	reti			;
;
;*******
;
;
TIM1_CAPT:			;
	in	TTEMP2,SREG	;Saves the status register

	;DEBUG
	;sbi	Pip_DDR,Pip_Bit	;Make it an output
	;sbi	Pip_Port,Pip_Bit;

TIM1_OFF:
	in	TTEMP,TIMSK	;Shut off this int
	andi	TTEMP,$F7	;
	out	TIMSK,TTEMP	;

TIM1_EXIT:
	;DEBUG
	;cbi	Pip_Port,Pip_Bit;

	out	SREG,TTEMP2	;restore the status register
	reti
;
;*********
;
TIM1_COMPA:			;
	in	TTEMP2,SREG	;Saves the status register

	;DEBUG
	;sbi	Pip_DDR,Pip_Bit	;Make it an output
	;sbi	Pip_Port,Pip_Bit;

TC1A_OFF:
	in	TTEMP,TIMSK	;
	andi	TTEMP,$BF	;
	out	TIMSK,TTEMP	;

TC1A_EXIT:

	;DEBUG
	;cbi	Pip_Port,Pip_Bit;

	out	SREG,TTEMP2	;restore the status register
	reti			;
;	
;********
;
;
TIM1_COMPB:		
	in	TTEMP2,SREG	;Saves the status register

	;DEBUG
	;sbi	Pip_DDR,Pip_Bit	;Make it an output
	;sbi	Pip_Port,Pip_Bit;

TC1B_OFF:
	in	TTEMP,TIMSK	;
	andi	TTEMP,$DF	;
	out	TIMSK,TTEMP	;

TC1B_EXIT:

	;DEBUG
	;cbi	Pip_Port,Pip_Bit;

	out	SREG,TTEMP2	;restore the status register
	reti			;
;
;*********
;
;Timer 1 gives us servo output timings from 1mS to 2mS, and allows
;the CPU to go do other things, rather than sitting there spinning 
;NOPS. It's a 16 bit counter, so I start it with a base value of servo_min,
;plus the servo width control byte (SERVO_X) We then have pulses that range
;from servo_min to servo_max with 8 bit resolution.
;
;Each servo out is activated sequentially in the main code. The activation
;code loads a value into Timer1, and sets the prescaler to /8, and sets the
;desired servo bit high. This int occurs when the servo's time expires, and
;we just shut off the output, and stop the timer. (Fast eh? :)
;
TIM1_OVF:
	in	TTEMP2,SREG	;Saves the status register

	;DEBUG
	;sbi	Pip_DDR,Pip_Bit	;Make it an output
	;sbi	Pip_Port,Pip_Bit;high

	;*******
	;Support for the R/C servo driver in SERVO.ASM
	;*******

	;No matter who was running, shut off all servo outputs
	ldi	TTEMP,$00	;
	out	PORTC,TTEMP	;

	in	TTEMP,TCCR1B	;Get the timer control register
	andi	TTEMP,$F8	;Set the prescaler to "off"
	out	TCCR1B,TTEMP	;Make it so.

	;This test could instead mask and look for a particular
	;bit to reduce the number of active servos.

	lds	TTEMP,Servo_Control	;

	;To make the servos run from 1 to 8, enable this code
	lsl	TTEMP			;Set next servo to run
	brcc	Servo_TB		;If no carry then done, else
	ldi	TTEMP,$01		;start over on servo 1(0)

	;To make the servos run from 8 to 1, enable this code
	;lsr	TTEMP			;Set next servo to run
	;brcc	Servo_TB		;If no carry then done, else
	;ldi	TTEMP,$80		;start over on servo 1(0)

	;If there was a ROL without carry, then I wouldn't have needed
	;to test for carry and reload like that. can't use LSL

Servo_TB:
	sts	Servo_Control,TTEMP	;Set the next active servo bit on.

TIM1O_EXIT:

	;DEBUG
	;cbi	Pip_Port,Pip_Bit;

	out	SREG,TTEMP2		;restore the status register
	reti				;
;
;*********
;
;	Main 1mS Opsys tick
;
; This seems like a huge routine, but actually, very little of it gets executed
; on any particular int. The main thing that we do, is decrement to zero any 
; timer bytes that aren't zero yet, and provide the state machine for the morse 
; code output that may have been started in MORSE.ASM
;
; There is also support for a timed wait between serial output chars, and 
; handlers for hardware or Xon/Xoff handshake.
;
TIM0_OVF:

	in	TTEMP3,SREG	;Saves the status register

	;DEBUG
	;sbi	Pip_DDR,Pip_Bit	;Make it an output
	;sbi	Pip_Port,Pip_Bit;Just high

	;**************
	;Reload timer 0 now to preserve accuracy
	;**************
	ldi	TTEMP,T0DIV	;Reload early, avoid the latency
	out	TCNT0,TTEMP	;

	;********
	;Keyboard debounce timer support
	;********
T0_Key	:
	lds	TTEMP,Key_Timer	;
	and	TTEMP,TTEMP	;Is it zero?
	breq	T0_Key_Done	;leave it alone if so
	dec	TTEMP		;Otherwise dec
	sts	Key_Timer,TTEMP	;and store
T0_Key_Done:	;Dummy label


	;********
	;Command Interpreter delay support
	;********
T0_CMD	:
	lds	TTEMP,CMDTIME	;
	and	TTEMP,TTEMP	;Is it zero?
	breq	T0_CMD_Done	;leave it alone if so
	dec	TTEMP		;Otherwise dec
	sts	CMDTIME,TTEMP	;and store
T0_CMD_Done:	;Dummy label

	;***********
	;EEPROM timer support
	;***********
T0_EEPROM:
	lds	TTEMP,EEPROM_Timer	;
	and	TTEMP,TTEMP	;Is it zero?
	breq	T0_EE_Done	;leave it alone if so
	dec	TTEMP		;Otherwise dec
	sts	EEPROM_Timer,TTEMP ;and store
T0_EE_Done:	;Dummy label

	;***********
	;LCD Display scroll timer support
	;***********
T0_LCD	:
	lds	TTEMP,LCD_Timer	;
	and	TTEMP,TTEMP	;Is it zero?
	breq	T0_LCD_Done	;leave it alone if so
	dec	TTEMP		;Otherwise dec
	sts	LCD_Timer,TTEMP	;and store
T0_LCD_Done:	;Dummy label

	;***********
	;VFD Display scroll timer support
	;***********
T0_VFD	:
	lds	TTEMP,VFD_Timer	;
	and	TTEMP,TTEMP	;Is it zero?
	breq	T0_VFD_Done	;leave it alone if so
	dec	TTEMP		;Otherwise dec
	sts	VFD_Timer,TTEMP	;and store
T0_VFD_Done:	;Dummy label

	;******
	;Morse code support
	;******
T0_Morse:

	;First, if we aren't active, then just ignore
	lds	TTEMP,BEEPFLAG	;Status check
	rol	TTEMP		;Active?
	brcc	T0_M_D		;If not, do something else.

	andi	TTEMP,$40	;Is the end flag set?
	breq	T0_Beep_State	;If not, go look at beeps

	;We get to here if beepflag is xx1xxxxx, meaning we are
	;doing the trailing silence

T0_End_Beep:
	lds	TTEMP,BEEPTIME	;Counting down
	dec	TTEMP		;One less
	sts	BEEPTIME,TTEMP	;Save it
	and	TTEMP,TTEMP	;Is it zero?
	brne	T0_M_D		;If not, just fly on
	
	lds	TTEMP,BEEPTIME2	;More to go?
	and	TTEMP,TTEMP	;Is it zero?
	breq	T0_End_Beep2	;If so, then we're done
	dec	TTEMP		;Now it's one less
	sts	BEEPTIME2,TTEMP	;Save it
	ldi	TTEMP,$FF	;and reload beeptime for another round
	sts	BEEPTIME,TTEMP	;
	rjmp	T0_M_D		;And bail, no sense checking further
				;until this one is empty

T0_End_Beep_A:
	lds	TTEMP,Dots_2_Go	;Any more "dots" to send?
	and	TTEMP,TTEMP	;
	breq	T0_End_Beep2	;Nope, done

	dec	TTEMP		;One less dot to send
	sts	Dots_2_Go,TTEMP	;Save that
	lds	TTEMP,M_SPEED	;Get a dot-time
	sts	BEEPTIME,TTEMP	;Reload the counter
	rjmp	T0_M_D		;Done

T0_End_Beep2:
	clr	TTEMP		;Flag it inactive
	sts	BEEPFLAG,TTEMP	;
	cbi	Morse_Port,Morse_Bit;Make sure of the exit state
	rjmp	T0_M_D		;

	;If we're in the middle of a beep or silent, then
	;dec and carry on. If we hit the end, then it's time
	;to change the state
T0_Beep_State:
	lds	TTEMP,BEEPTIME	;
	dec	TTEMP		;
	sts	BEEPTIME,TTEMP	;
	brne	T0_M_D		;If non-zero, then we're done

	;If zero:
	lds	TTEMP,Dots_2_Go	;Any more "dots" to send?
	and	TTEMP,TTEMP	;
	breq	T0_M_Change_State ;Nope, done

	dec	TTEMP		;One less dot to send
	sts	Dots_2_Go,TTEMP	;Save that
	lds	TTEMP,M_SPEED	;Get a dot-time
	sts	BEEPTIME,TTEMP	;Reload the counter
	rjmp	T0_M_D ;Done

	;At the beginning, the count is zero, and the state
	;is beep. 
	;Each time we come through, we fetch another bit-pair
	;which indicates dot (short beep), dash (long beep)
	;or end. 
	;For each bit pair, we do a beep of either one or three
	;units long, and we do a one unit silence.
	;If the state is beep, and the fetched value is end,
	;then we set bit 7 to 0, and exit

T0_M_D: rjmp	T0_M_Done	;Trick for long rel jmp

T0_M_Change_State:
	lds	TTEMP,BEEPFLAG	;
	mov	TTEMP2,TTEMP	;A copy
	andi	TTEMP,$40	;
	breq	T0_M_Beep	;

T0_M_Sil:
	mov	TTEMP,TTEMP2	;fresh copy
	andi	TTEMP,$BF	;Reset silent bit
	ori	TTEMP,$80	;Set active bit
	sts	BEEPFLAG,TTEMP	;
	cbi	Morse_Port,Morse_Bit;Output silent
	lds	TTEMP,M_SPEED	;Inter-symbol silence is
	sts	BEEPTIME,TTEMP	;always one unit
	ldi	TTEMP,0		;
	sts	Dots_2_Go,TTEMP	;
	rjmp	T0_M_Done	;We're done, till the silence
				;is over
T0_M_Beep:
	lds	TTEMP,BEEPFLAG	;
	inc	TTEMP		;Next State
	andi	TTEMP,$07	;
	ori	TTEMP,$C0	;Set flag bit and active bit
	sts	BEEPFLAG,TTEMP	;
	sbi	Morse_Port,Morse_Bit;output beep

T0_M_Lookup:
	;TTEMP has 0-7 now
	;and I need to know what di-bit in MBITS and MBITS2
	;correspond
	;
	mov	TTEMP2,TTEMP	;copy
	andi	TTEMP2,$04	;high bit only
	breq	T0_Get_MB	;

T0_Get_MB2:
	lds	TTEMP2,MBITS2	;
	rjmp	T0_M_Bits	;

T0_Get_MB:
	lds	TTEMP2,MBITS1	;

;A long, but fast strip of the appropriate bits.
	
T0_M_Bits:
	andi	TTEMP,$03	;Pick the right bit-pair
	breq	T0_M_B0		;0-3 = AA BB CC DD
	cpi	TTEMP,$01	;One
	breq	T0_M_B1		;
	cpi	TTEMP,$02	;Two
	breq	T0_M_B2		;
	rjmp	T0_M_B3		;Three

T0_M_B3:
	andi	TTEMP2,$03	;Mask
	breq	T0_M_Dit	;Handle 00 (*)
	cpi	TTEMP2,$01	;
	breq	T0_M_Dah	;Handle 01 (-)
	rjmp	T0_M_End	;Handle END

T0_M_B2:
	andi	TTEMP2,$0C	;Mask
	breq	T0_M_Dit	;Handle 00 (*)
	cpi	TTEMP2,$04	;
	breq	T0_M_Dah	;Handle 01 (-)
	rjmp	T0_M_End	;Handle END

T0_M_B1:
	andi	TTEMP2,$30	;
	breq	T0_M_Dit	;Handle 00 (*)
	cpi	TTEMP2,$10	;
	breq	T0_M_Dah	;Handle 01 (-)
	rjmp	T0_M_End	;Handle END

T0_M_B0:
	andi	TTEMP2,$C0	;
	breq	T0_M_Dit	;Handle 00 (*)
	cpi	TTEMP2,$40	;
	breq	T0_M_Dah	;Handle 01 (-)
	rjmp	T0_M_End	;Handle END

T0_M_Dit:
	lds	TTEMP,M_SPEED	;get the time reload
	sts	BEEPTIME,TTEMP	;save it
	ldi	TTEMP,0		;
	sts	BEEPTIME2,TTEMP	;
	sts	Dots_2_Go,TTEMP	;
	rjmp	T0_M_Done	;

T0_M_Dah:
	lds	TTEMP,M_SPEED	;get the time reload
	sts	BEEPTIME,TTEMP	;save it
	ldi	TTEMP,0		;
	sts	BEEPTIME2,TTEMP	;
	ldi	TTEMP,2		;
	sts	Dots_2_Go,TTEMP	;
	rjmp	T0_M_Done	;

T0_M_End:
	lds	TTEMP,BEEPFLAG	;
	andi	TTEMP,$BF	;Reset the beeping flag
	ori	TTEMP,$20	;Flag for three units of silence
	sts	BEEPFLAG,TTEMP	;

	lds	TTEMP,M_SPEED	;get the time reload
	sts	BEEPTIME,TTEMP	;
	ldi	TTEMP,0		;
	sts	BEEPTIME2,TTEMP	;

	ldi	TTEMP,1		;
	sts	Dots_2_Go,TTEMP	;
	cbi	Morse_Port,Morse_Bit;Output a silence
	rjmp	T0_M_Done	;	

T0_M_Done:	;Dummy label, fallthrough

	;*******
	;Support for the stepper motor handler
	;*******

T0_Step:lds	TTEMP,Step_Time	;Get delay timer from SRAM
	and	TTEMP,TTEMP	;Is it zero?
	breq	T0_Step_Done	;leave it alone if so
	dec	TTEMP		;Otherwise dec
	sts	Step_Time,TTEMP	;and store
T0_Step_Done:

	;*******
	;Support for the timed watchdog handler
	;*******

T0_Dog:	lds	TTEMP,Dog_Food	;Get delay timer from SRAM
	and	TTEMP,TTEMP	;Is it zero?
	breq	T0_Dog_Done	;leave it alone if so
	dec	TTEMP		;Otherwise dec
	sts	Dog_Food,TTEMP	;and store
T0_Dog_Done:

	;*******
	;Support for the R/C servo driver in SERVO.ASM
	;*******

T0_RC:	lds	TTEMP,Frame_Delay;Get delay timer from SRAM
	and	TTEMP,TTEMP	;Is it zero?
	breq	T0_RC_Done	;leave it alone if so
	dec	TTEMP		;Otherwise dec
	sts	Frame_Delay,TTEMP;and store

T0_RC_Done: ;Dummy Label

	;*******
	;Support for the timed wait for charachter
	;*******
T0_Char:lds	TTEMP,Char_Time	;Get delay timer from SRAM
	and	TTEMP,TTEMP	;Is it zero?
	breq	T0_Char_Done	;leave it alone if so
	dec	TTEMP		;Otherwise dec
	sts	Char_Time,TTEMP	;
T0_Char_Done: ;Dummy Label



T0_Ser_Rx_Hs:
	lds	TTEMP,Serial_State	;
	andi	TTEMP,$08		;Were we saying SHADDAP?
	breq	T0_Ser			;If no, go here

	;Ok, we must have been full at one time, are we now?
	lds	TTEMP3,Serial_In_Size	;
	cpi	TTEMP3,(Serial_In_Buffer_Length - Serial_In_Threshold)
	brge	T0_Ser			;If GE, then we're still full

	andi	TTEMP,$20		;
	brne	T0_Ser_RX_Xon		;

T0_Ser_Rx_Hw:
	;Set hardware HS off, and let's talk
	cbi	SER_HS_OUTPUT,Ser_Hs_Out_Pin	;Set hardware HS low
	lds	TTEMP,Serial_State	;
	andi	TTEMP,$F7		;Reset RX Shaddap flag
	sts	Serial_State,TTEMP	;
	rjmp	T0_Ser

T0_Ser_Rx_Xon:
	;Jam an Xon, and let them talk
	lds	TTEMP,Serial_State	;
	andi	TTEMP,$F4		;Clear the Xon/Xoff bits
					;and the RX Shaddap flag
	ori	TTEMP,$01		;Tell it to say Xon
	sts	Serial_State,TTEMP	;
T0_SER:	

	;!!! Later, drop a conditional here, so that we
	;can skip these tests if XON/XOFF isn't being used,
	;according to a bit in SERIAL_STATE
	lds	TTEMP,Serial_State	;
	andi	TTEMP,$30		;Is any handshake in use?
	breq	T0_Ser_Char_Delay	;If no, go here

	cpi	TTEMP,$20		;Is it Xon?
	breq	T0_Ser_Xoff		;If so, go here
					;Else it must be hardware

	;Test for inbound handshake.
	;If asserted, then don't reload the uart until it's
	;de-asserted.
	cbi	SER_HS_IN_DDR,Ser_HS_IN_Pin	;Set as input
	nop					;Wait for sync
	sbic	SER_HS_INPUT,Ser_HS_IN_Pin	;
	rjmp	T0_Ser_Done		;If hardware HS has us blocked
	rjmp	T0_Ser_Char_Delay	;

	;Are we Xoffed?
T0_Ser_Xoff:
	lds	TTEMP,Serial_State	;Get the status
	andi	TTEMP,$40		;If X1XXXXXX then we have been
	brne	T0_Ser_Done		;Xoffed, so stay quiet

	;*********
	;Support for the inter-char delay feature in SERIAL.ASM
	;*********
	;Since the Char_Delay is  non-zero, then dec and exit
T0_Ser_Char_Delay:
	lds	TTEMP,Char_Delay	;Get delay timer from SRAM
	and	TTEMP,TTEMP		;Is it zero?
	breq	T0_Ser_Char_Delay_Done	;leave it alone if so
	dec	TTEMP			;else dec, and if NOW zero
	sts	Char_Delay,TTEMP	;Put it away
	brne	T0_Ser_Done		;If still non-zero, then don't
					;send, regardless of HS
T0_Ser_Char_Delay_Done:


;	Else check for stuff to send
T0_Ser_GotStuff:
	lds	TTEMP,Serial_Out_Size	;Get the size
	and	TTEMP,TTEMP		;Is it zero?
	breq	T0_Ser_Done		;If so, exit with UDRIE Off

	;Not Handshaking, not Xoffed, Not delaying, and Got stuff to send
	;So GET ON WITH IT!
	sbi	UCR,UDRIE		;Re-enable UDRIE

T0_Ser_Done:

TIM0_EXIT:

	;DEBUG
	;cbi	Pip_Port,Pip_Bit;

	out	SREG,TTEMP3	;restore the status register
	reti			;Bye!
;
;*********
;
;SPI is unused at this point
;
SPI_Handler:			;
	in	TTEMP2,SREG	;Saves the status register

	;DEBUG
	;sbi	Pip_DDR,Pip_Bit	;Make it an output
	;sbi	Pip_Port,Pip_Bit;

SPI_OFF:
	ldi	TTEMP,$3F	;Disable int, disable SPI
	out	SPCR,TTEMP	;

SPI_EXIT:
	;DEBUG
	;cbi	Pip_Port,Pip_Bit;

	out	SREG,TTEMP2	;restore the status register
	reti			;
;
;***************
;
;Take a char and put it in the Serial input buffer. 
;Takes about 8uS due to handshaking logic
;
;
UART_RXC:			;
	in	TTEMP2,SREG	;Saves the status register

	;DEBUG
	;sbi	Pip_DDR,Pip_Bit	;Make it an output
	;sbi	Pip_Port,Pip_Bit;Two pings then high

	;***
	;Loading incoming UART data into the serial input buffer
	;***

	in	TTEMP,UDR	;Get the char

	;Test for XON/XOFF handshake in operation
	lds	TTEMP3,Serial_State	;
	andi	TTEMP3,$20		;Is Xon handshake in use?
	breq	Uart_RXC_Store		;If not, just go store it.

Uart_Rxc_T4_Xon:
	cpi	TTEMP,XON		;If this is an Xon, then
	brne	Uart_Rxc_T4_Xoff	;Fix it so the transmitter
	lds	TTEMP,Serial_State	;can talk
	andi	TTEMP,$3F		;
	sts	Serial_State,TTEMP	;
	rjmp	UART_RXC_Exit		;

Uart_Rxc_T4_Xoff:		
	cpi	TTEMP,Xoff		;If this is an XOFF, then
	brne	Uart_Rxc_Store		;set the state so we stop
	ori	TTemp,$C0		;talking, also eat the XOFF char
	sts	Serial_State,TTEMP	;
	rjmp	UART_RXC_Exit		;
	
Uart_Rxc_Store:

	push	YH		;
	push	YL		;
	push	TEMP2		;

 	ldi	YL,low(Serial_In_Buffer)	;Point at the buffer
	ldi	YH,High(Serial_In_Buffer)	;
	ldi	TEMP2,Serial_In_Buffer_Length	;

	ld	TTEMP3,Y		;Get the size
	cp	TEMP2,TTEMP3		;Compare to the length
	breq	Uart_RXC_Exit		;If full, then nothing to do.
	inc	TTEMP3			;increment that, since we stored something
	st	Y,TTEMP3		;and put it away

	ldd	TTEMP3,Y+2		;Get the tail pointer
	inc	TTEMP3			;Bump it
	cp	TTEMP3,TEMP2		;Did it wrap? If Temp > TEMP2, then yes
	brlo	UART_RXC_Put_Go		;If less than
	breq	UART_RXC_Put_Go		;Or equal, it's ok
	ldi	TTEMP3,1		;An incremented zero!

Uart_RXC_Put_Go:			;It's a go, let's do it.
	std	Y+2,TTEMP3		;Still pointing to the size

	add	YL,TTEMP3		;Add the tail pointer to the base
	adc	YH,ZERO			;ZERO is always zero, but carry isn't
	std	Y+2,TTEMP		;The offset gives us data
	rjmp	Store_Serial_In_HS	;

Store_Serial_In_HS:

	lds	TTEMP3,Serial_State	;Get the serial state
	andi	TTEMP3,$08		;Are we already in "Shaddap"?
	brne	UART_RXC_Exit		;if so, then just exit.

	;Check if we should say XOFF (or do the handshake)
	lds	TTEMP3,Serial_In_Size	;
	cpi	TTEMP3,(Serial_In_Buffer_Length-Serial_in_Threshold);Running out of room?
	brlo	UART_RXC_Exit		;If not, then just exit

	;We get here, if the receive buffer is getting full, and
	;we need to do some handshaking.

	lds	TTEMP3,Serial_State	;Find out what mode is set
	cpi	TTEMP3,$20		;Is it Xoff?
	breq	Uart_RXC_Say_Xoff	;If so, go do that
	cpi	TTEMP3,$10		;Is it hardware?
	breq	Uart_RXC_Say_HSOFF	;If so, go do that
	rjmp	UART_RXC_Exit		;Otherwise, store the char
					;and continue, and hope he
					;shuts up real soon!
Uart_RXC_Say_Xoff:
	lds	TTEMP3,Serial_State	;
	ori	TTEMP3,2		;Signal to jam an Xoff
	sts	Serial_State,TTEMP3	;
	sbi	UCR,UDRIE		;and allow talking, just in case
	rjmp	UART_RXC_Shaddap	;

Uart_RXC_Say_HSOFF:
	sbi	SER_HS_OUTPUT,Ser_Hs_Out_Pin	;Set hardware HS high
	
Uart_RXC_Shaddap:
	lds	TTEMP3,Serial_State		;
	ori	TTEMP3,$08			;
	sts	Serial_State,TTEMP3		;Mark that we are in SHADDAP

UART_RXC_Exit:
	pop	TEMP2		;
	pop	YL		;
	pop	YH		;

	;DEBUG
	;cbi	Pip_Port,Pip_Bit;

	out	sreg,TTEMP2	;restore the status register
	reti			;
;
;***************************************************************
;
;Ready to transmit the next byte
;
;This takes about 10uS at 8 MHz, because of the handshake logic
;
UART_DRE:			;
	in	TTEMP2,SREG	;Saves the status register

	;DEBUG
	;sbi	Pip_DDR,Pip_Bit	;Make it an output
	;sbi	Pip_Port,Pip_Bit;Signal beginning of ISR


	lds	TTEMP,Serial_State	;Get the serial status byte
	andi	TTEMP,$30		;Is any handshaking in use?
	breq	UART_DRE_BUFFER_Check	;If not, just see what to send


UART_DRE_Hardware_HS:
	cpi	TTEMP,$10		;Is it hardware?
	brne	UART_DRE_XOFF_Test	;Nope, must be Xoff

	;Test for inbound handshake.
	;If asserted, then don't reload the uart until it's
	;de-asserted.
	cbi	SER_HS_IN_DDR,Ser_HS_IN_Pin	;Set as input
	nop					;Wait for sync
	sbic	SER_HS_INPUT,Ser_HS_IN_Pin	;
	rjmp	Uart_DRE_HS		;If hardware HS has us blocked, 
					;then just exit
	rjmp	UART_DRE_Buffer_Check	;Otherwise, go see if we need to
					;say something

UART_DRE_XOFF_Test:
	;test for Xoff handshake	;
	;If an Xoff was recvd, then Serial_State = X1XXXXXX
	;until an XON is received
	lds	TTEMP,Serial_State	;
	andi	TTEMP,$40		;Test with mask
	brne	Uart_DRE_Xoff		;

	;Check if we need to jam XON or XOFF in the uart output as
	;a priority override on the next char.
	lds	TTEMP,Serial_State	;Get the flag
	andi	TTEMP,$03		;Is it zero?
	breq	UART_DRE_TX		;If nothing, then ignore
	cpi	TTEMP,1			;If 1 then send Xon
	breq	GSO_Xon			;Else Xoff

GSO_Xoff:
	ldi	TTEMP,Xoff		;
	rjmp	Get_Serial_Out_Jam	;

GSO_Xon:
	ldi	TTEMP,Xon		;
	rjmp	Get_Serial_Out_Jam	;

Get_Serial_Out_Jam:
	lds	TTEMP2,Serial_State	;
	andi	TTEMP2,$80		;Clear the flag so we don't loop
	sts	Serial_State,TTEMP2	;
	rjmp	UDRE_TX_Send		;Since this is a jam, don't mess
					;with the buffer


UART_DRE_BUFFER_Check:
	lds	TTEMP,Serial_Out_Size	;Get the current size
	and	TTEMP,TTEMP		;
	brne	UART_DRE_TX	;If non-zero, then let it continue
	
	cbi	UCR,UDRE	;
	rjmp	UART_DRE_Empty	;	

UART_DRE_TX:

	Push    YH      	;
        push    YL		;
        push    TEMP2   	;
        push    TEMP		;	

	ldi	YL,low(Serial_Out_Buffer)	;Point at the buffer
	ldi	YH,High(Serial_Out_Buffer)	;
	ldi	TEMP2,Serial_Out_Buffer_Length	;and get it's length (constant)

Uart_DRE_Get:
	ld	TEMP,Y			;Get the size (now pointing at the head)
	dec	TEMP			;
	st	Y,TEMP			;
	ldd	TEMP,Y+1		;Get the head pointer
	inc	TEMP			;Bump it
	cp	TEMP,TEMP2		;Will it wrap?
	brlo	Uart_DRE_Get_Go_A	;If less than, then no
	breq	Uart_DRE_Get_Go_A	;If equal then no.
	ldi	TEMP,1			;an incremented zero

Uart_DRE_Get_Go_A:
	std	Y+1,TEMP		;Put the head pointer back
	add	YL,TEMP			;
	adc	YH,ZERO			;Now pointing at the data
	ldd	TTEMP,Y+2		;Get the data

Uart_DRE_Get_Done:
	pop	TEMP		;
	pop	TEMP2		;
	pop	YL		;
	pop	YH		;
	;
	; Time------------>
	;  S L X X X X X M S S
	;  T S X X X X X S P T 
	;  A B X X X X X B C P
	;1 0 0 0 0 0 0 0 0 0 1  (Line state)
	;
	;Comment both of the following out for 8N1
UDRE_TX_Send:
	andi	TTEMP,$7F		;7 bit Space parity
	;ori	TTEMP,$80		;7 bit Mark  parity
	out	UDR,TTEMP		;Load uart from TEMP

	ldi	TTEMP,SER_DLY		;Inter-char delay
	sts	Char_Delay,TTEMP	;Reload the delay
	and	TTEMP,TTEMP		;But if it's zero
	breq	Uart_DRE_Exit		;Then don't turn off UDRE!

	;disable Uart_DRE_int.
	cbi	UCR,UDRE		;
	rjmp	Uart_DRE_Exit		;


UART_DRE_HS:
	cbi	UCR,UDRE		;Not talking any more
	rjmp	UART_DRE_EXIT		;

UART_DRE_EMPTY:
	rjmp	UART_DRE_EXIT		;

UART_DRE_XOFF:
	rjmp	UART_DRE_EXIT		;

UART_DRE_EXIT:

	;Debug
	;cbi	Pip_Port,Pip_Bit;
	
	out	SREG,TTEMP2	;restore the status register
	reti			;
;
;***************************************************************
;
;Uart TXC is unused at this point
;
UART_TXC:
	in	TTEMP2,SREG	;Saves the status register


	;DEBUG
	;sbi	Pip_DDR,Pip_Bit	;Make it an output
	;sbi	Pip_Port,Pip_Bit;


UTXC_OFF:
	in	TTEMP,UCR	;
	andi	TTEMP,$BF	;Disable the TXC int.
	out	UCR,TTEMP	;

UTXC_EXIT:
	;DEBUG
	;cbi	Pip_Port,Pip_Bit;

	out	SREG,TTEMP2	;restore the status register
	reti			;
;
;***************************************************************
;
;Analog comparator is unused at this point.
;
ANA_COMP:
	in	TTEMP2,SREG	;Saves the status register

	;DEBUG
	;sbi	Pip_DDR,Pip_Bit	;Make it an output
	;sbi	Pip_Port,Pip_Bit;

ANAC_OFF:
	ldi	TTEMP,$83	;Shut off power to the comparator
	out	ACSR,TTEMP	;and the comparator ints

ANAC_EXIT:

	;DEBUG
	;cbi	Pip_Port,Pip_Bit;

	out	SREG,TTEMP2	;restore the status register
	reti
