;***************************************************************************
; 
; File Name		:'Interp.asm"
; Title			:Simple tokenized command interpreter
; Date			:03/31/2000
; Version		:0.02
; Support telephone	:765 287 1987  David B. VanHorn
; Support fax		:765 287 1989
; Support Email		:dvanhorn@cedar.net
; Support Snail		;1104 E 13th St, Muncie IN 47302
; Target MCU		:AT90S8515
; 
;***************************************************************************;
;	D E S C R I P T I O N
;
;Simple tokenized command interpreter
;Just the basics for now, Delay, Action, Loop, Skip(+/-)N commands.
;Later, I'll add some variables, and math on them, tests and such
;
;Note that you could easily hack this over into interpreting commands stored
;in rom. I'm not sure WHY, but hey, it's an option.
;
;IMPORTANT NOTE:  There is absolutely NO error checking here. If you code 
;a skip -200 in a three byte program, good luck! If you fail to include a 
;parameter, then things are going to immediately get VERY wierd.
;Adding sanity checks to all this would take space in EEPROM, and time
;to execute, and would make this example needlessly complicated.
;
;FWIW: There are a few million systems out there in the world, that handle
;your money every day, that have a similar implementation of a language 
;interpreter. They use a loader to strip out comments, but the programming
;language is things like B1.1GA17, and if you ever leave out the semicolon
;that denotes comments, they will cheerfully execute your comments.
;(Ask me how I know.)  Anyway, my point is that you don't always need a
;GUI development environment with heuristic syntax checking and code 
;optimization.
;
;***************************************************************************;
;	M O D I F I C A T I O N   H I S T O R Y 
;
;
;       rev.      date    who   why
;	----	--------  ---	------------------------------------------
;	0.01	00.03.28  dvh	Creation
;	0.02	00.03.31  dvh	First three commands, plus basic structure is
;				working. Skip is the hard one, and we can skip
;				forward, backward, or zero (infinite loop)
;				It was easier to implement here, than in the F84
;				that's for sure!
;				Command zero: MS_Wait, 1 byte parm
;				Command one:  Set Servo X, two bytes, first is the
;				servo number to set (0-7) and the second is the 
;				position, 0-255
;				Command two: Immediate Skip, 1 byte parm 
;				00 = infinite loop
;				01 = +1 (essentially a NOP, kinda pointless)
;				FF = -1
;				
;				Fill in some more commands here.
; 
;				Command Last: Loop (go back to the beginning)
;				
; 
;	See the end of this routine for a test program, set up to load
;	into EEPROM
;
;
;********************************************************************
;
.DSEG
.equ	cmd_start_low = $20	;Below this used for settings store/recall
.equ	cmd_start_high = $00	;Program memory can only extend 255 bytes up
;				;from this due to single byte math, but can
;				;cross pages, (16 bit pointer)
;
;Any or all of these could be in registers to speed this up.
;
CMDLOW:		.byte	1	;Low byte of Pointer to EEPROM for command data
CMDHIGH:	.byte	1	;High byte
CMDTIME:	.byte	1	;The waiting time for CMD0, dec'd by T0OV
CMDAT:		.byte	1	;Which command are we currently pointing at
CMDWANT:	.byte	1	;Which one do we want to execute next
.CSEG
;
;********************************************************************
;
Interp:

	;A command that wants to wait, must set CMDTIME to the proper number
	;of millisec, and also set CMDWAIT so we know someone's waiting

	;First, see if there's a command timeout running.
	;If so, then just return to the OS
	;
	lds	TEMP,CMDTIME	;Are we waiting for something?
	and	TEMP,TEMP	;Is it zero?
	breq	Interpreter	;If zero, then we can proceed
	ret			;Otherwise, still waiting, return to system
	;
	;Takes all of 500nS to loop thru, then returns to the OS

;****************************************************************************
;Now we need to know what command we are supposed to execute. This is done
;with a pair of pointer bytes, CMDAT and CMDWANT. The meanings should be 
;obvious. If CMDWANT = CMDAT then the current command is executed. 
;Otherwise, we skip through the EEPROM, reading commands and skipping data 
;till they become equal.  Each command is responsible for skipping it's own 
;parameters and must exit with CMDLOW and CMDHIGH pointing at the EEPROM 
;address of the next command.
;The only exception is LOOP, which restores initial states to CMDLOW,CMDHIGH,
;CMDWAIT, CMDAT, and CMDWANT.
;
Interpreter:

	ldi	TEMP,$20	;Always begin at the beginning
	sts	CMDLOW,TEMP	;
	ldi	TEMP,$00	;
	sts	CMDHIGH,TEMP	;

	sts	CMDAT,TEMP	;We always scan from the bottom
	rcall	CMD_Scan	;
	ret			;This return really never gets executed until
				;something executes, see below.

;WARNING!! A bit of trickiness employed here.
;A command that executes (as opposed to scan) will pop the stack twice,
;and return to the RET above, and then back to the system.

CMD_Scan:
	rcall	Get_EE		;Get the command byte

	;Test the command byte, and see which routine we should execute.
	;A jump table would be faster if lots of commands are available
	;but this is smaller for now.
	
	;A compare for every existing command

Int_cmd_0:
	cpi	TEMP,$00	;Is it command zero?
	brne	INT_CMD_1	;If not, maybe command one?
	rcall	CMD_MS_Wait	;If so, then call the command, but if
				;it executes, it will NOT return to here!

	rjmp	CMD_Scan	;If it returns to here, then we didn't 
				;execute, then we need to keep scanning
				;Scanned commands move the EEPROM address
				;pointer in CMDHIGH and CMDLOW 
INT_CMD_1:
	cpi	TEMP,$01	;One entry per valid command
	brne	INT_CMD_2	;See above for comments.
	rcall	CMD_Set_Servo	;
	rjmp	CMD_Scan	;

INT_CMD_2:
	cpi	TEMP,$02	;
	brne	INT_CMD_3	;
	rcall	CMD_Skip_N	;
	rjmp	CMD_Scan	;

	;Add additional commands here, and provide handlers below.

INT_CMD_3:

	;Add them starting at 3, and work down. That way, any unimplemented
	;commands fall through to loop, which is a sort-of non-fatal error handler.
	;Alternately, you could put loop as command 3, and make an error handler for
	;command 255. It's up to you!


	;All invalid commands end up here, which is essentially a
	;"warm boot"
INT_CMD_255:
	rjmp	CMD_Loop	;And a handler in case we get an
				;un-implemented command, which is
				;also the "loop" command

;Each command knows what it's parameters should look like, and it moves the 
;command pointer to point at the next command by the act of reading it's 
;parameters.

;****************************************************************************
;The parameter is a single byte, indicating the number of mS to wait.
;NOTE that this is not a dumb loop, we return to the OS, and execution
;will resume after N mS expires. Meanwhile code in main, and ISRs all 
;execute normally
;
CMD_MS_Wait:
	lds	TEMP,CMDAT	;Which command are we on?
	lds	TEMP2,CMDWANT	;Which one do we want
	cp	TEMP,TEMP2	;Same?
	breq	CMD_MS_Wait_Exec;If so, execute
	inc	TEMP		;
	sts	CMDAT,TEMP	;
	ldi	TEMP,1		;
	rcall	Eat_Parm	;Just moves the EEPROM pointer
	ret			;This cmd always has a single byte parm

CMD_MS_Wait_Exec:
	rcall	Get_EE		;Get the parameter
	sts	CMDTIME,TEMP	;Say how long to wait
	lds	TEMP,CMDWANT	;Point to the next command
	inc	TEMP		;
	sts	CMDWANT,TEMP	;
	pop	TEMP		;Break out of the scan
	pop	TEMP		;
	ret			;All done

;****************************************************************************
;Command one will set servos to some position
;The parameter is two bytes, the servo to set, (0-7)
;and the position to set it at. (0-255)
;
CMD_Set_Servo:	

	lds	TEMP,CMDAT	;Which command are we on?
	lds	TEMP2,CMDWANT	;Which one do we want
	cp	TEMP,TEMP2	;Same?
	breq	CMD_Servo_Exec	;If so, execute
	inc	TEMP		;
	sts	CMDAT,TEMP	;
	ldi	TEMP,2		;Just moves the EEPROM pointer
	rcall	Eat_Parm	;This one always has two bytes
	ret	

CMD_Servo_Exec:
	rcall	Get_EE		;Get which servo
	andi	TEMP,$07	;Mask off to prevent pointing outside
				;the servo memory accidentally.

	ldi	ZL,low(Servo_0)	;Point at the servo memory
	ldi	ZH,high(Servo_0);
	add	ZL,TEMP		;Point at the desired servo
	adc	ZH,ZERO		;Note that zero is the first servo

	rcall	Get_EE		;Get servo position
	st	Z,TEMP		;Store the new value.
				;The OS will update the servo the next time
				;it's turn comes up
	lds	TEMP,CMDWANT	;Point to the next command
	inc	TEMP		;
	sts	CMDWANT,TEMP	;
	pop	TEMP		;Break out of the scan
	pop	TEMP		;
	ret			;

;****************************************************************************
;Explicit skip command
;Skips forward or backward, within the program
;Warning, Skip zero is an infinite loop.
;If you skip outside the program, then god help you..
;
;This is a little tricky, in that we have no idea how long the
;command parameters are, if there are any at all!
;The only way to resolve this, is to "scan" through the commands
;and let the commands "eat" the memory, and drop us in the right 
;place.
;
;CMDPOINTER tells us what command we are on right now.
;The fetched value tells us what command we want to be on.
;(as in CMDPOINTER + (VALUE))
;
CMD_Skip_N:
	lds	TEMP,CMDAT	;Which command are we on?
	lds	TEMP2,CMDWANT	;Which one do we want
	cp	TEMP,TEMP2	;Same?
	breq	CMD_Skip_Exec	;If so, execute
	inc	TEMP		;
	sts	CMDAT,TEMP	;
	ldi	TEMP,1		;
	rcall	Eat_Parm	;Just moves the EEPROM pointer
	ret	

CMD_Skip_Exec:
	rcall	Get_EE		;How many to skip (limited to +/- 127)
	lds	TEMP2,CMDAT	;Get the pointer to the next command
	add	TEMP2,TEMP	;Add the intended offset
	sts	CMDWANT,TEMP2	;Store this, so next command seek will look for
				;the command at current + offset (offset can be 
				;negative, FF = -1)
	pop	TEMP		;Break out of the scan
	pop	TEMP		;
	ret			;

;****************************************************************************
;The last command is the loop command, which also serves as a 
;catch-all for any unimplemented commands. There is no parameter,
;It simply restores the command interpreter to the initial state.
;This is also called from INIT, to set up the entry conditions.
;

CMD_Loop:
	lds	TEMP,CMDAT	;Which command are we on?
	lds	TEMP2,CMDWANT	;Which one do we want
	cp	TEMP,TEMP2	;Same?
	breq	CMD_Loop_Exec	;If so, execute
	inc	TEMP		;
	sts	CMDAT,TEMP	;
	ldi	TEMP,1		;
	rcall	Eat_Parm	;Just moves the extraction poniter
	ret	

CMD_Loop_Exec:
	rcall	CMD_INIT	;Init the interpreter
	pop	TEMP		;Break out of the scan
	pop	TEMP		;
	ret			;

;****************************************************************************
;Used at init, and when LOOP is called.
;
CMD_INIT:
	ldi	TEMP,cmd_start_low
	sts	CMDLOW,TEMP	;
	ldi	TEMP,cmd_start_high
	sts	CMDHIGH,TEMP	;
	
	ldi	TEMP,0		;
	sts	CMDTIME,TEMP	;No timeout pending (JIC)
	sts	CMDWANT,TEMP	;Reset command counter
	sts	CMDAT,TEMP	;and desired command both to zero
	ret		
;****************************************************************************
;
;Generic routine used to read the current byte, and point at the next.
;
Get_EE:
	lds	XL,CMDLOW	;Pointing at the command
	lds	XH,CMDHIGH	;
	rcall	Read_EE		;Get the byte and inc X
	sts	CMDLOW,XL	;
	sts	CMDHIGH,XH	;
	ret			;
;
;****************************************************************************
;Skips the number of bytes in TEMP. Slightly faster than recalling EEPROM
;
Eat_Parm:
	lds	XL,CMDLOW	;
	lds	XH,CMDHIGH	;
	add	XL,TEMP		;Skip the parm memory
	adc	XH,ZERO		;
	sts	CMDLOW,XL	;Store the absolute pointer
	sts	CMDHIGH,XH	;
	ret			;

;********************************************************************
;
;This is how you would put a "factory default" program in EEPROM
;
.eseg	;Put this in EEPROM
.org	$020			;Make sure that you have this same value above in
				;cmd_start_low and cmd_start_high!

	.db	0		;Command Wait N mS (first byte is ALWAYS a command)
	.db	255		;How many mS to wait
	.db	1		;Command Set Servo
	.db	0		;servo zero 
	.db	200		;which position
	.db	0		;Command Wait
	.db	255		;255mS
	.db	1		;Command Set Servo
	.db	0		;Servo zero
	.db	32		;to 32
	.db	3		;Command Loop

	.db	3		;Loop (JIC)

.cseg	;Back to code