;***************************************************************************
; 
; File Name		:'CIRCSER.asm"
; Title			:
; Date			:
; Version		:
; Support telephone	:765 287 1987  David B. VanHorn
; Support fax		:765 287 1989
; Support Email		:dvanhorn@cedar.net
; Support Snail		;1104 E 13th St, Muncie IN 47302
; Target MCU		:AT90S8515
; 
;***************************************************************************;
;	D E S C R I P T I O N
;
;Circular serial I/O buffers, with handshake handling for both hardware 
;and/or Xon-Xoff  Thanks to Jack Tidwell for the inspiration, but the bugs
;are mine :)
;
;There is a bitflag in Serial_State for Xon/Xoff and hardware handshake
;The ISR code in UDRIE and TXCIE handle the handshaking, all you have to do 
;here is select which (if any) you want.
;
;Also, in Xon mode, you might want to send an Xon if the serial buffer is
;empty, in case the other party thinks you said Xoff a while back. Then 
;again, you might not. It's all up to you.
;
;If you've never done a circular buffer, this bears some explanation.
;There's really no "circularity" about it, it's just another linear byte 
;array. The magic is in the pointers.  You have a head and tail pointer.
;In this implementation, I have a constant that tells me the size of the 
;array, and a variable that stores the number of bytes stored. This saves
;me from having to do ugly math on the pointers.
;
;First, take an array of memory locations:
;123456789ABCDEF  The size is a variable, and the length is a constant (15)
;I set the head and tail pointers to the same location, say "A" (it could be
;anywhere) Since the head is pointing at the tail, the buffer is empty (Or full!)
;Regardless of whatever data might be in there. I would also set the size 
;variable to zero. This is how we know the buffer is empty.
;
;Now I store some bytes, and the tail pointer moves twoard F. If it reaches F, 
;then the next byte must be stored at 1, and you have to handle the "wrap". This
;is the "circular" part. Also, you have to check that the head and tail pointer
;are not equal after the store, as this would mean that the buffer is full.
;Having the external variable "size" allows us to distinguish H-T=Full from 
;H=T=Empty
;
;One thing to note, is that the buffer store routine is only concerned with the
;tail pointer, and the recall routine is only concerned with the head pointer.
;Only the buffer init routine is concerned with both pointers!
;
;***************************************************************************;
;	M O D I F I C A T I O N   H I S T O R Y 
;
;
;       rev.      date    who   why
;	----	--------  ---	------------------------------------------
;	0.01	99.09.13  dvh	Creation
;	0.02	99.09.17  dvh	All working, with hardware handshake.
;				Xon/Xoff not tested yet.
;				What fun!
;	0.03	00.03.28  dvh	Added Scan_Serial_In to detect when a
;				terminating char arrives.
;******************************************************************
;Hardware
;******************************************************************
;
;These pins don't have to be on the same port at all
;
.CSEG
.equ	SER_HS_OUTPUT=PORTA	;Hardware Handshake output port
.equ	SER_HS_OUT_DDR=DDRA	;Hardware Handshake output control
.equ	Ser_HS_Out_Pin=2	;Hardware Handshake Output bit

.equ	SER_HS_INPUT=PINA	;Hardware Handshake input port
.equ	SER_HS_IN_DDR=DDRA	;Hardware Handshake input control
.equ	Ser_HS_IN_Pin=3		;Hardware Handshake Input bit
;
;******************************************************************
;Software
;******************************************************************
;
;You have to define things, before you can use them.
;Below, in the CSEG (Code Segment) I define some constants.
;Then, in the DSEG (Data segment) I use them to allocate some memory.
;If you reverse this order, the assembler may not pick up an error,
;But you will experience "non-deterministic execution".
;
.CSEG
.equ	SER_DLY=0		        ;Inter-Charachter delay in Milliseconds
.equ	Serial_In_Threshold=2	        ;Now many bytes free before we panic
.equ	Serial_In_Buffer_Length = 50	;Any length, not necessarily
.equ	Serial_Out_Buffer_Length = 50	;equal

.DSEG
Watch_Char:	.byte	1	;Char to watch for
Char_Time:	.byte	1	;How long to watch for it in mS, dec'd by TO ISR
Char_Delay:	.byte	1	;Delay between chars, in mS dec'd by TO ISR
Ser_Eat_Timer:	.byte	1	;<<<DEBUG>>>  Every N mS eat a char from idle

Serial_State:		.byte 1	;1XXXXXXX  TX in Shaddap state
				;0XXXXXXX  TX in "Talking"
				;
				;X1XXXXXX  Xoff asserted, pause transmit
				;X0XXXXXX  Xon asserted, let 'er rip
				;XX1XXXXX  Using Xon/Xoff Handshaking
				;XX0XXXXX  Not Using Xon/Xoff Handshaking
				;
				;XXX1XXXX  Using hardware Handshaking
				;XXX0XXXX  Not Using hardware Handshaking
				;XXXX1XXX  RX in Shaddap state
				;XXXX0XXX  RX in "Talk to me"
				;
				;XXXXXX10  Need to jam an Xoff into TX
				;XXXXXX01  Need to jam an XON into TX
				;XXXXXX00  No action
				;XXXXXX11  No action

;Note: the way this is coded, the buffers can be anywhere, but MUST
;be <= 255 bytes long.

Serial_In_Buffer:	
Serial_In_Size:		.byte	1
Serial_In_Head:		.byte	1
Serial_In_Tail:		.byte	1
Serial_In_Data:		.byte	Serial_In_Buffer_Length

Serial_Out_Buffer:	
Serial_Out_Size:	.byte	1
Serial_Out_Head:	.byte	1
Serial_Out_Tail:	.byte	1
Serial_Out_Data:	.byte	Serial_Out_Buffer_Length

.CSEG
;
;**********************************************************************
;
;Set up the inital states for the buffers, and make sure the handshake
;is turned on (Not the Shaddap state), so we can talk.
;
Init_Serial:
	ldi	TEMP,0
	sts	Serial_In_Head,TEMP	;Nothing in here!
	sts	Serial_In_Tail,TEMP	;
	sts	Serial_In_Size,TEMP	;

	sts	Serial_Out_Head,TEMP	;Nothing in here either!
	sts	Serial_Out_Tail,TEMP	;
	sts	Serial_Out_Size,TEMP	;

	sts	Char_Delay,TEMP		;Signal watch not active
        sts     Watch_Char,TEMP         ;Just keeping it neat

	;ldi	TEMP,$20		;Enable Xon/Xoff handshake
	ldi	TEMP,$10		;Enable hardware handshake
	sts	Serial_State,TEMP	;Ok, you can talk to me.

	;Set up Hardware HS I/O bits
	sbi	SER_HS_OUT_DDR,Ser_HS_Out_Pin	;Set as output
	cbi	SER_HS_IN_DDR,Ser_HS_IN_Pin	;Set as input

	;Assert hardware HS to off		;
	cbi	SER_HS_OUTPUT,Ser_Hs_Out_Pin	;Set low

	;Should we send xon?
	;rcall	HS_Xon	;Jam XON into serial output buffer
	;NOTE: This may not always be advisable, you need to know
	;what you're talking to.
	;

	rcall	Set_Baud_9600	;Or something else if you prefer

	cbi	UCR,CHR9	;No 9 bit funny stuff
	sbi	UCR,RXCIE	;Enable the RX complete interrupt
	cbi	UCR,TXCIE	;Disable the transmit complete int.
	sbi	UCR,RXEN	;Enable Uart Receive (takes pin D0)
	sbi	UCR,TXEN	;Enable Uart Transmit (takes pin D1)

	;Transmit ints are left disabled until a char is put in the
	;Serial_Out_Buffer

	ret
;
;
;************************************************************************************
;
Ser_TEST1:
	ldi	ZL,low (Signon_Message*2) ;Point at a signon message (in TABLES.ASM)
	ldi	ZH,high (Signon_Message*2) ;
	rcall	Store_Ser_Out_Buf	;Won't store anything that won't fit!
	ret
;
;************************************************************************************

Signon_Message:
;
;The first one is too long to fit the current serial out buffer, so it would never get sent.
;Useful to check the buffer guard logic though!
;
;	 00000000011111111112222222222333    3          3       3          3        3  3  3  
;	 12345678901234567890123456789012    3          4       5          6        7  8  9
;.db	"AVR8515 Startup program version ","MAJOR_REV",".","Minor_Rev","Minor_RevB",CR,LF,NUL
.db	"The Quick Brown Fox Jumps Over",CR,LF,0
;.db	"ABCDEFGHIJKLMNOPQRSTUVWXYZ123456",0


;
;**********************************************************************
;
;Assumes Z pointing to the string in question, which is null-terminated
;Checks for suficient space in the output buffer, and does not store 
;anything, if the entire string won't fit.
;NEVER send strings longer than the buffer, they will never go!
;
Store_Ser_Out_Buf:
	ldi	TEMP,Serial_Out_Buffer_Length;(const)Len-Size=Freespace
	lds	TEMP2,Serial_Out_Size	;(Variable)
	clc				;
	sub	TEMP,TEMP2		;
	brcs	SSO_D			;If full, then just exit

SSO_E:
	Push	ZH			;
	push	ZL			;
	clr	TEMP2			;Ready to count

	;Get size of message		;
SSO_Loop:
	lpm				;Get the data
	ld	TEMP3,Z+		;Get a char, and toss it.
	inc	TEMP2			;Count a char
	and	R0,R0			;Is it a null?
	brne	SSO_Loop		;Nope, try again

	pop	ZL			;
	pop	ZH			;
	dec	TEMP2			;Compensate for the null
	
	clc				;
	sub	TEMP,TEMP2		;Temp has the current 
	brcs	SSO_D			;If it won't fit, then exit

SSO_Send_Loop:
	lpm				;
	and	R0,R0			;
	breq	SSO_Done		;
	ld	TEMP,Z+			;
	mov	TEMP,R0			;Now that we know it will fit,
	rcall	Store_Serial_Out	;we can just load it into the 
	rjmp	SSO_Send_Loop		;output buffer

SSO_D:	nop

SSO_Done:
	ret

;
;*********************************************************************
;
;Gets a char from the RX circular buffer (if there is one!) and signals
;success by setting the carry flag. If carry is zero, then we didn't get
;anything.
;
Get_Serial_In:

	ldi	YL,low(Serial_In_Buffer)	;Point at the buffer
	ldi	YH,High(Serial_In_Buffer)	;
	ldi	TEMP2,Serial_In_Buffer_Length	;and get it's length (constant)
	rcall	Circ_Get			;Get a char, if there is one.
	brcc	Get_Serial_In_Done_Char		;If there's nothing in there, then
	

Get_Serial_In_Done_Char:
	ret	;passing the carry flag and data through to the calling routine
;
;*********************************************************************
;
;Look through the serial in buffer, and see if the char in TEMP is present
;in the active part of the buffer.
;Sets carry flag if the char is present
;
Scan_Serial_In:
	ldi	YL,low(Serial_In_Buffer)	;Point at the buffer
	ldi	YH,High(Serial_In_Buffer)	;
	lds	LOOP,Serial_In_Buffer_Length	;and get it's length (constant)
	and	LOOP,LOOP			;If the buffer is empty
	breq	Scan_Serial_In_Fail		;then obviously we're done

Scan_Serial_Loop:
	dec	LOOP				;One less to scan
	breq	Scan_Serial_In_Fail		;If done, then exit, failing
	ld	TEMP2,Y+			;Get the byte, and bump the pointer
	cp	TEMP2,TEMP			;Does it match?
	brne	Scan_Serial_Loop		;If not, scan again

Scan_Serial_In_Success:
	sec					;Indicate failure
	ret					;

Scan_Serial_In_Fail:				;Indicate Success
	clc					;
	ret
;
;*********************************************************************
;
;Puts a char in the TX circular buffer, if there's room.
;If no room, we exit with carry cleared, if stored, we exit with carry set.
;
Store_Serial_Out: 

	;Data is in TEMP!
	ldi	YL,low(Serial_Out_Buffer)	;Point at the buffer
	ldi	YH,High(Serial_Out_Buffer)	;
	ldi	TEMP2,Serial_Out_Buffer_Length	;and get it's length (constant)
	rcall	Circ_Put			;Get a char, if there is one.
	brcc	Store_Serial_Out_Done		;

Store_Serial_Out_Done:
	ret					;Byte in TEMP,carry set or not

;
;********************************************************************
;
;Tom Hubrin's Method:
;From the data sheet:BAUD = XTAL/(16*(UBRR+1))
;Rearranging you get:UBRR = XTAL/(16*BAUD) - 1
;
;This would work except that any fraction will be truncated. 
;For least error we want to round off the result. 
;So we add 1/2 as follows: UBRR = XTAL/(16*BAUD) - 1 + 0.5
;Combining the -1 and the +0.5 we can rewrite this as follows:
;
;UBRR = XTAL/(16*BAUD) - 0.5
;
;The problem is that we cannot work with fractions in 
;integer math. But we can rearrange it so that the end 
;result is the same without using any decimal points.
;
;UBRR = XTAL/(16*BAUD) - 0.5 * (16*BAUD)/(16*BAUD)
;UBRR = XTAL/(16*BAUD) - (8*BAUD)/(16*BAUD)
;So: UBRR = ( XTAL - 8*BAUD )/( 16*BAUD )
;
;Note that we are NOT doing any of this in the machine.
;The assembler simply uses the most recently declared value
;of BaudRate to evaluate the expression. The result is
;really "ldi TEMP,N" where N is the right divisor for the
;desired baud rate.

;Can't get lower than 2400 with an 8 MHZ xtal

Set_Baud_2400:
	.set BaudRate=2400
	ldi TEMP, ( XTAL - 8*BaudRate ) / ( 16*BaudRate )
	rcall SET_BAUD
	ret

Set_Baud_4800:
	.set BaudRate=4800
	ldi TEMP, ( XTAL - 8*BaudRate ) / ( 16*BaudRate )
	rcall	SET_BAUD	;
	ret

Set_Baud_9600:
	.set BaudRate=9600
	ldi TEMP, ( XTAL - 8*BaudRate ) / ( 16*BaudRate )
	rcall	SET_BAUD	;
	ret

Set_Baud_14400:
	.set BaudRate=14400
	ldi TEMP, ( XTAL - 8*BaudRate ) / ( 16*BaudRate )
	rcall	SET_BAUD	;
	ret

Set_Baud_19200:
	.set BaudRate=19200
	ldi TEMP, ( XTAL - 8*BaudRate ) / ( 16*BaudRate )
	rcall	SET_BAUD	;
	ret

Set_Baud_28800:
	.set BaudRate=28800
	ldi TEMP, ( XTAL - 8*BaudRate ) / ( 16*BaudRate )
	rcall	SET_BAUD	;
	ret

Set_Baud_57600:
	.set BaudRate=57600
	ldi TEMP, ( XTAL - 8*BaudRate ) / ( 16*BaudRate )
	rcall	SET_BAUD	;
	ret

Set_Baud_115200:
	.set BaudRate=115200
	ldi TEMP, ( XTAL - 8*BaudRate ) / ( 16*BaudRate )
	rcall	SET_BAUD	;
	ret

Set_Baud:

	mov	LOOP,TEMP	;Save it for a moment
	rcall	WT4TXMT		;Wait till all prev sent data is gone
	out	UBRR,LOOP	;Set the baud rate
	ret			;
;
;************************************************************
;
;This one waits for TX buffer empty
;
WT4TXMT:
	rcall	Timed_Smack		;Reset the watchdog if needed
	lds	TEMP,Serial_Out_Tail	;Where is the head?
	lds	TEMP2,Serial_Out_Head	;Where is the tail?
	cp	TEMP,TEMP2		;If not equal, then the buffer
	brne	WT4TXMT			;is full. 
	ret
;
;This one waits till the current char is done transmitting
;
WT4XCHR:
	rcall	Timed_Smack	;Reset watchdog if needed
	in	TEMP,USR	;
	andi	TEMP,$40	;Get the TX Complete bit
	breq	WT4XCHR		;
	ret
;
;************************************************************	
;
SEND_BREAK:

	;Assumption, that you want anything placed in the serial buffer
	;to be sent before the break. 
	rcall	WT4TXMT		;Wait for serial buffer empty
	rcall	WT4XCHR		;Wait for the last char to get out

	;Shift the baud rate slightly lower.
	in	TEMP,UBRR	;Get the current baud rate (51 @ 9600)
	push	TEMP		;Save it

	ldi	TEMP,255	;Much simpler, works for most baud rates.
	out	UBRR,TEMP	;

	ldi	TEMP,NUL	;Load a null, 8 bits of zero
	out	UDR,TEMP	;Start Breaking now.	

	rcall	WT4XCHR		;Wait till the char is sent

	pop	TEMP		;Get the old baud rate
	out	UBRR,TEMP	;Restore the original baud rate

	ret
;
;************************************************************
;
;Input: Char to get in temp, Delay in TEMP2
;Output: TEMP2=0 for ok, or FF for timeout or wrong char
;
;WT4CHAR: ;This starts the Serial_Wait engine to look for an incoming char.
;	
;	sts	Char_Time,TEMP2	;System will decrement on 1mS system ticks
;	sts	Watch_Char,TEMP	;save temp for compare
;	cbi	UCR,RXCIE	;Turn off serial in ints.
;	rcall	Serial_Wait	;
;	sbi	UCR,RXCIE	;Turn the serial ints back on
;	ret			;;
;
;Char_Time = 0, Not waiting, or timed out.
;Char_Time <> 0, actively waiting.
;Watch_Char = byte value waiting on.
;


;Serial_Wait:
;	lds	TEMP,Char_Time	;Are we waiting on a char (0=no)
;	and	TEMP,TEMP	;Is Char_Time=0?
;	brne	WT4CHAR_ACTIVE	;If not, then we're waiting
;	ret			;Else return

	;If we're waiting, then see if it's happened

WT4CHAR_ACTIVE:
	in	TEMP,USR	;Did we get a char yet?
	andi	TEMP,$80	;Check bit 7
	brne	WT4CHAR_GOTIT	;
	ret			;Nope, just keep waiting

WT4CHAR_GOTIT:

	in	TEMP2,UDR	;Get the Char
	andi	TEMP2,$7F	;Mask off parity(?)
	lds	TEMP,Watch_Char	;What were we looking for?
	cp	TEMP,TEMP2	;Is this it?
	breq	WT4CHAR_SUCCESS	;

WT4CHAR_SUCCESS:
	ret



